% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CDnetsim.R
\name{simcdnet}
\alias{simcdnet}
\title{Simulate data from Count Data Model with Social Interactions}
\usage{
simcdnet(
  formula,
  contextual,
  Glist,
  theta,
  deltabar,
  delta = NULL,
  rho = 0,
  tol = 1e-10,
  maxit = 500,
  data
)
}
\arguments{
\item{formula}{an object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be as for example \code{y ~ x1 + x2 | x1 + x2}
where \code{y} is the endogenous vector, the listed variables before the pipe, \code{x1}, \code{x2} are the individual exogenous variables and
the listed variables after the pipe, \code{x1}, \code{x2} are the contextual observable variables. Other formulas may be
\code{y ~ x1 + x2} for the model without contextual effects, \code{y ~ -1 + x1 + x2 | x1 + x2} for the model
without intercept or \code{y ~ x1 + x2 | x2 + x3} to allow the contextual variable to be different from the individual variables.}

\item{contextual}{(optional) logical; if true, this means that all individual variables will be set as contextual variables. Set the
\code{formula} as \code{y ~ x1 + x2} and \code{contextual} as \code{TRUE} is equivalent to set the formula as \code{y ~ x1 + x2 | x1 + x2}.}

\item{Glist}{the adjacency matrix or list sub-adjacency matrix.}

\item{theta}{the true value of the vector \eqn{\theta = (\lambda, \beta', \gamma')'}. The parameter \eqn{\gamma} should be removed if the model
does not contain contextual effects (see details).}

\item{deltabar}{the true value of \eqn{\bar{\delta}}{deltabar}.}

\item{delta}{the true value of the vector \eqn{\delta = (\delta_2, ..., \delta_{\bar{R}})}{\delta = (\delta_2, ..., \delta_{Rbar})}. If \code{NULL}, then \eqn{\bar{R}}{Rbar} is set to one and \code{delta} is empty.}

\item{rho}{the true value of \eqn{\rho}.}

\item{tol}{the tolerance value used in the Fixed Point Iteration Method to compute the expectancy of \code{y}. The process stops if the \eqn{L_1}{L1} distance
between two consecutive values of the expectancy of \code{y} is less than \code{tol}.}

\item{maxit}{the maximal number of iterations in the Fixed Point Iteration Method.}

\item{data}{an optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{mcmcARD} is called.}
}
\value{
A list consisting of:
\item{yst}{ys (see details), the latent variable.}
\item{y}{the observed count data.}
\item{yb}{ybar (see details), the expectation of y.}
\item{Gyb}{the average of the expectation of y among friends.}
\item{marg.effects}{the marginal effects.}
\item{rho}{the return value of rho.}
\item{Rmax}{infinite sums in the marginal effects are approximated by sums up to Rmax.}
\item{iteration}{number of iterations performed by sub-network in the Fixed Point Iteration Method.}
}
\description{
\code{simcdnet} is used simulate counting data with rational expectations (see details). The model is presented in Houndetoungan (2022).
}
\details{
Following Houndetoungan (2022), the count data \eqn{\mathbf{y}}{y} is generated from a latent variable \eqn{\mathbf{y}^*}{ys}.
The latent variable is given for all i as
\deqn{y_i^* = \lambda \mathbf{g}_i \mathbf{E}(\bar{\mathbf{y}}|\mathbf{X},\mathbf{G})  + \mathbf{x}_i'\beta + \mathbf{g}_i\mathbf{X}\gamma + \epsilon_i,}{ys_i = \lambda g_i*E(ybar|X, G) + x_i'\beta + g_i*X\gamma + \epsilon_i,}
where \eqn{\epsilon_i \sim N(0, 1)}{\epsilon_i --> N(0, 1)}.\cr
Then, \eqn{y_i = r} iff \eqn{a_r \leq y_i^* \leq a_{r+1}}{a_r \le ys_i \le a_{r + 1}}, where
\eqn{a_0 = -\inf}{a_0 = -Inf}, \eqn{a_1 = 0}, \eqn{a_r = \sum_{k = 1}^r\delta_k}{a_r = \delta_1 + ... + \delta_r}.
The parameter are subject to the constraints \eqn{\delta_r \geq \lambda}{\delta_r \ge \lambda} if \eqn{1 \leq r \leq \bar{R}}{1 \le r \le Rbar},  and
\eqn{\delta_r = (r - \bar{R})^{\rho}\bar{\delta} + \lambda}{a_r = deltabar*(r - Rbar)^{\rho} + \lambda} if \eqn{r \geq \bar{R} + 1}{r \ge Rbar + 1}.
}
\examples{
\donttest{
# Groups' size
M      <- 5 # Number of sub-groups
nvec   <- round(runif(M, 100, 1000))
n      <- sum(nvec)

# Parameters
lambda <- 0.4
beta   <- c(1.5, 2.2, -0.9)
gamma  <- c(1.5, -1.2)
delta  <- c(1, 0.87, 0.75, 0.6)
delbar <- 0.05
theta  <- c(lambda, beta, gamma)

# X
X      <- cbind(rnorm(n, 1, 1), rexp(n, 0.4))

# Network
Glist  <- list()

for (m in 1:M) {
  nm           <- nvec[m]
  Gm           <- matrix(0, nm, nm)
  max_d        <- 30
  for (i in 1:nm) {
    tmp        <- sample((1:nm)[-i], sample(0:max_d, 1))
    Gm[i, tmp] <- 1
  }
  rs           <- rowSums(Gm); rs[rs == 0] <- 1
  Gm           <- Gm/rs
  Glist[[m]]   <- Gm
}


# data
data    <- data.frame(x1 = X[,1], x2 =  X[,2])

rm(list = ls()[!(ls() \%in\% c("Glist", "data", "theta", "delta", "delbar"))])

ytmp    <- simcdnet(formula = ~ x1 + x2 | x1 + x2, Glist = Glist, theta = theta, 
                    deltabar = delbar, delta = delta, rho = 0, data = data)

y       <- ytmp$y

# plot histogram
hist(y, breaks = max(y))}
}
\references{
Houndetoungan, E. A. (2022). Count Data Models with Social Interactions under Rational Expectations. Available at SSRN 3721250, \doi{10.2139/ssrn.3721250}.
}
\seealso{
\code{\link{cdnet}}, \code{\link{simsart}}, \code{\link{simsar}}.
}
