% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PredictionContributions.R
\name{prediction_contributions_data}
\alias{prediction_contributions_data}
\title{Model term contributions to predicted response}
\usage{
prediction_contributions_data(
  data,
  model = NULL,
  coefficients = NULL,
  coeff_cols = NULL,
  vcov = NULL,
  add_var = list(),
  groups = list(),
  conf.level = 0.95,
  interval = c("confidence", "prediction", "none"),
  bar_labs = rownames(data)
)
}
\arguments{
\item{data}{A user-defined data-frame containing values for compositional variables
along with any additional variables that the user wishes to predict for.
If left blank, a selection of observations (2 from each level of
richness) from the original data used to fit the model would be selected.}

\item{model}{A Diversity Interactions model object fit by using the
\code{\link[DImodels:DI]{DI()}} function from the
\code{\link[DImodels:DImodels-package]{DImodels}} package.}

\item{coefficients}{If a regression model is not available (or can't be fit in R),
the regression coefficients from a model fit in some other
language can be used to calculate predictions. However, the
user would have to ensure there's an appropriate one-to-one
positional mapping between the data columns and the
coefficient values. Further, they would also have to provide
a variance-covariance matrix of the coefficients in the `vcov`
parameter if they want the associated CI for the prediction or
it would not be possible to calculate confidence/prediction
intervals using this method.}

\item{coeff_cols}{If `coefficients` are specified and a one-to-one positional
mapping between the data-columns and coefficient vector is
not present. A character string or numeric index can be specified
here to reorder the data columns and match the corresponding
coefficient value to the respective data column. See the
"Use model coefficients for prediction" section in examples.}

\item{vcov}{If regression coefficients are specified, then the variance-covariance
matrix of the coefficients can be specified here to calculate the
associated confidence interval around each prediction. Failure to do
so would result in no confidence intervals being returned. Ensure
`coefficients` and `vcov` have the same positional mapping with the data.}

\item{add_var}{A list specifying values for additional predictor variables
in the model independent of the compositional predictor variables.
This could be useful for comparing the predictions across
different values for a non-compositional variable.
If specified as a list, it will be expanded to show a plot
for each unique combination of values specified, while if specified
as a data-frame, one plot would be generated for each row in the
data and they will be arranged in a grid according to the
value specified in `nrow` and `ncol`.}

\item{groups}{A list specifying groupings to arrange coefficients into.
The coefficients within a group will be added together and
shown as a single component on the respective bars in the plot.
This could be useful for grouping multiple similar terms
into a single term for better visibility.}

\item{conf.level}{The confidence level for calculating confidence or
prediction intervals.}

\item{interval}{Type of interval to calculate:
\describe{
  \item{"none"}{No interval to be calculated.}
  \item{"confidence" (default)}{Calculate a confidence interval.}
  \item{"prediction"}{Calculate a prediction interval.}
}}

\item{bar_labs}{The labels to be shown for each bar in the plot. The user
has three options:
   - By default, the row-names in the data would be used as
     labels for the bars.
   - A character string or numeric index indicating an ID
     column in data.
   - A character vector of same length as the number of rows
     in the data, which manually specifies the names for each bar.
If none of the three options are available, the function would
assign a unique ID for each bar.}
}
\value{
A data-frame with the following columns. Any additional columns which
weren't used when fitting the model would also be present.
 \describe{
   \item{.Community}{An identifier column to discern each
                     observation in the data. These are the labels which
                     will be displayed for the bars in the plot.}
   \item{.add_str_ID}{An identifier column for grouping the cartesian product
                      of all additional columns specified in `add_var`
                      parameter (if `add_var` is specified).}
   \item{.Pred}{The predicted repsonse for each observation.}
   \item{.Lower}{The lower limit of the prediction interval for
                 each observation.}
   \item{.Upper}{The lower limit of the prediction interval for
                 each observation.}
   \item{.Contributions}{An identifier describing the name of the
                         coefficient contributing to the response.}
   \item{.Value}{The contributed value of the respective coefficient/group
                 to the total prediction.}
 }
}
\description{
The helper function for preparing the data to split the predicted response
from a regression model into contributions (predictor coefficient * predictor value)
by the terms in the model. The output of this function can be passed to the
`\link{prediction_contributions_plot}` function to visualise the results.
}
\examples{
library(DImodels)
library(dplyr)

## Load data
data(sim2)

## Fit model
mod <- glm(response ~ 0 + (p1 + p2 + p3 + p4)^2, data = sim2)

prediction_contributions_data(data = sim2[c(1,5,9,11), ],
                              model = mod)

## Specific coefficients can also be grouped together
## Either by their indices in the model coefficient vector
prediction_contributions_data(data = sim2[c(1,5,9,11), ],
                              model = mod,
                              groups = list("Interactions" = 5:10))
## Or by specifying the coefficient names as character strings
prediction_contributions_data(data = sim2[c(1,5,9,11), ],
                              model = mod,
                              groups = list("p1_Ints" = c("p1:p2",
                                                          "p1:p3",
                                                          "p1:p4")))

## Additional variables can also be added to the data by either specifying
## them directly in the `data` or by using the `add_var` argument
## Refit model
sim2$block <- as.numeric(sim2$block)
new_mod <- update(mod, ~. + block, data = sim2)
## This model has block so we can either specify block in the data
subset_data <- sim2[c(1,5,9,11), 2:6]
subset_data
head(prediction_contributions_data(data = subset_data,
                                   model = new_mod))
## Or we could add the variable using `add_var`
subset_data <- sim2[c(1,5,9,11), 3:6]
subset_data
head(prediction_contributions_data(data = subset_data,
                                   model = new_mod,
                                   add_var = list(block = c(1, 2))))
## The benefit of specifying the variable this way is we have an ID
## columns now called `.add_str_ID` which would be used to create a
## separate plot for each value of the additional variable


## Model coefficients can also be used, but then user would have
## to specify the data with all columns corresponding to each coefficient
coef_data <- sim2 \%>\%
               mutate(`p1:p2` = p1*p2, `p1:p3` = p1*p2, `p1:p4` = p1*p4,
                      `p2:p3` = p2*p3, `p2:p4` = p2*p4, `p3:p4` = p3*p4) \%>\%
               select(p1, p2, p3, p4,
                      `p1:p2`, `p1:p3`, `p1:p4`,
                      `p2:p3`, `p2:p4`, `p3:p4`) \%>\%
               slice(1,5,9,11)
print(coef_data)
print(mod$coefficients)
prediction_contributions_data(data = coef_data,
                              coefficients = mod$coefficients,
                              interval = "none")
## To get uncertainity using coefficients vcov matrix would have to specified
prediction_contributions_data(data = coef_data,
                              coefficients = mod$coefficients,
                              vcov = vcov(mod))

## Specifying `bar_labs`
## Our data has four rows so we'd need four labels in bar_labs
prediction_contributions_data(data = coef_data,
                              coefficients = mod$coefficients,
                              vcov = vcov(mod),
                              bar_labs = c("p1 Domm", "p2 Domm",
                                           "p3 Domm", "p4 Domm"))
}
