% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dmcData.R
\name{dmcObservedData}
\alias{dmcObservedData}
\title{dmcObservedData}
\usage{
dmcObservedData(
  dat,
  stepCAF = 20,
  stepDelta = 5,
  outlier = c(200, 1200),
  columns = c("VP", "Comp", "RT", "Error"),
  compCoding = c("comp", "incomp"),
  errorCoding = c(0, 1),
  quantileType = 5,
  delim = "\\t",
  skip = 0
)
}
\arguments{
\item{dat}{Text file(s) containing the observed data or an R DataFrame (see createDF/addDataDF)}

\item{stepCAF}{Step size for the CAF bins. For example, a step size of 20 would result
in 5 CAF bins centered on 10, 30, 50, 70, and 90\%.}

\item{stepDelta}{Step size for the Delta bins. For example, a step size of 5 would result
in 19 CAF bins positioned at 5, 10, 15, ... 85, 90, 95\%.}

\item{outlier}{Outlier limits in ms (e.g., c(200, 1200))}

\item{columns}{Name of required columns DEFAULT = c("VP", "Comp", "RT", "Error")}

\item{compCoding}{Coding for compatibility DEFAULT = c("comp", "incomp")}

\item{errorCoding}{Coding for errors DEFAULT = c(0, 1))}

\item{quantileType}{Argument (1-9) from R function quantile specifying the algorithm (?quantile)}

\item{delim}{Single character used to separate fields within a record if reading from external text file.}

\item{skip}{Number of lines to skip before reading data if reading from external text file.}
}
\value{
DataFrame
}
\description{
Basic example analysis script to create data object required
for observed data. Example raw *.txt files are flankerData.txt and simonData.txt. There
are four critical columns:
A column containing participant number
A column coding for compatible or incompatible
A column with RT (in ms)
A column indicating of the response was correct
}
\examples{
# Example 1
plot(flankerData)  # flanker data from Ulrich et al. (2015)

# Example 2
plot(simonData)    # simon data from Ulrich et al. (2015)

# Example 3 (Basic behavioural analysis from Ulrich et al. 2015)
flankerDat <- tibble::add_column(Task = "flanker", flankerData$summaryVP, .before = 2)
simonDat   <- tibble::add_column(Task = "simon",   simonData$summaryVP,   .before = 2)
datAgg     <- rbind(rbind(flankerDat, simonDat))

datAgg$VP   <- factor(datAgg$VP)
datAgg$Task <- factor(datAgg$Task)
datAgg$Comp <- factor(datAgg$Comp)

aovErr <- aov(perErr ~ Comp*Task + Error(VP/(Comp*Task)), datAgg)
summary(aovErr)
model.tables(aovErr, type = "mean")

aovRt <- aov(rtCor ~ Comp*Task + Error(VP/(Comp*Task)), datAgg)
summary(aovRt)
model.tables(aovRt, type = "mean")

# Example 4
dat <- createDF(nVP = 50, nTrl = 500, design = list("Comp" = c("comp", "incomp")))
dat <- addDataDF(dat,
                 RT = list("Comp_comp"    = c(500, 75, 120),
                           "Comp_incomp"  = c(530, 75, 100)),
                 Error = list("Comp_comp" = c(3, 2, 2, 1, 1),
                            "Comp_incomp" = c(21, 3, 2, 1, 1)))
datOb <- dmcObservedData(dat)
plot(datOb)
plot(datOb, VP = 1)

# Example 5
dat <- createDF(nVP = 50, nTrl = 500, design = list("Congruency" = c("cong", "incong")))
dat <- addDataDF(dat,
                 RT = list("Congruency_cong"   = c(500, 75, 100),
                           "Congruency_incong" = c(530, 100, 110)),
                 Error = list("Congruency_cong"   = c(3, 2, 2, 1, 1),
                              "Congruency_incong" = c(21, 3, 2, 1, 1)))
datOb <- dmcObservedData(dat, stepCAF = 20, stepDelta = 10,
                         columns = c("VP", "Congruency", "RT", "Error"),
                         compCoding = c("cong", "incong"))
plot(datOb, labels = c("Congruent", "Incongruent"))
plot(datOb, VP = 1)
}
