\name{lhs.design}
\alias{lhs.design}
\alias{lhs.augment}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Functions for accessing latin hypercube sampling designs from package lhs }
\description{
  Functions for comfortably accessing latin hypercube sampling designs from package lhs, 
  which are useful for quantitative factors with many possible levels. In particular, 
  they can be used in computer experiments. The designs are random samples.
}
\usage{
lhs.design(nruns, nfactors, type="optimum", factor.names=NULL, seed=NULL, digits=NULL, ...)
lhs.augment(lhs, m=1, type="optAugment", seed=NULL, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{nruns}{ number of runs in the latin hypercube sample }
  \item{nfactors}{ number of factors in the latin hypercube sample }
  \item{type}{ character string indicating the type of design or augmentation method;
        defaults are \dQuote{optimum} for \code{lhs.design} and \dQuote{optAugment} 
        for \code{lhs.augment};\cr
        both functions call the functions named typeLHS from package 
        \pkg{lhs} in order to fulfill their task; possible types for \code{lhs.design} are 
        \code{genetic}, \code{improved}, \code{maximin}, \code{optimum}, \code{random}, 
        possible types for \code{lhs.augment} are \code{augment}, \code{optSeeded}, or \code{optAugment}.\cr 
        see the respective functions from package \pkg{\link[lhs]{lhs}} }
  \item{seed}{ seed for random number generation; latin hypercube samples from package 
        \pkg{lhs} are random samples. Specifying a seed makes the result reproducible. }
  \item{factor.names}{ list of scale end values for each factor; 
        names are used as variable names; if the list is not named, the variable names are X1, X2 and so forth;
        the original unit cube calculated by package \pkg{lhs} (scale ends 0 and 1 for each variable) 
        is rescaled to the values given in factor.names }
  \item{digits}{ digits to which the design columns are rounded; one single value 
        (the same for all factors) or a vector of length \code{nfactors};\cr
        note that the rounding is applied after generation of the design on the actual 
        data scale, i.e. the unit cube generated by the functions from package \pkg{lhs}
        is NOT rounded}
  \item{lhs}{design generated by function \code{lhs.design} (class \code{design}, 
        of type \code{lhs}}
  \item{m}{integer number of additional points to add to design \code{lhs} (note, however, 
        that \code{optSeeded} does not necessarily preserve all original runs!)}
  \item{\dots}{ additional arguments to the functions from package \pkg{lhs}; 
        refer to their documentation (functions for generating lhs designs: 
        \code{\link[lhs]{randomLHS}}, \code{\link[lhs]{geneticLHS}}, 
        \code{\link[lhs]{improvedLHS}}, \code{\link[lhs]{maximinLHS}}, 
        \code{\link[lhs]{optimumLHS}}; functions for augmenting lhs designs: 
        \code{\link[lhs]{augmentLHS}}, 
        \code{\link[lhs]{optSeededLHS}}, 
        \code{\link[lhs]{optAugmentLHS}})
        }
}
\details{
   Function \code{lhs.design} creates a latin hypercube sample, 
   function \code{lhs.augment} augments an existing latin hypercube sample (or in case of 
   type \code{optSeeded} takes the existing sample as the starting point but potentially modifies it). 
   In comparison to direct usage of package \pkg{lhs}, the functions add the possibility 
   of recoding lhs samples to a desired range, and they embed the lhs designs into 
   class \code{\link[DoE.base]{design}}.\cr
   Range coding is based on the recoding facility from package \pkg{rsm} and the 
   \code{factor.names} parameter used analogously to packages \pkg{DoE.base} and \pkg{FrF2}.
   
   The lhs designs are useful for quantitative factors, if it is considered desirable to uniformly 
   distribute design points over a hyperrectangular space. This is e.g. considered interesting 
   for computer experiments, where replications of the same settings are often useless.
   }
\value{
    Both functions return a data frame of S3 class \code{\link[DoE.base]{design}} 
    with attributes attached. 
    The data frame contains the experimental settings as recoded to the scale ends defined in factor.names (if given), 
    rounded to the number of digits given in \code{digits} (if given).
    The experimental factors in the matrix \code{desnum} attached as attribute \code{desnum} contain the 
    design in the unit cube (all experimental factors ranging from 0 to 1) as 
    returned by package \code{lhs}. \cr
    Function \code{lhs.augment} preserves additional variables (e.g. responses) that 
    have been added to the design \code{lhs} before augmenting. Note, however, that 
    the response data are NOT used in deciding about which points to augment the design with. 
    
    The attribute \code{run.order} is not very useful for this type of design, as there is no standard order. 
    It therefore is present for formal reasons only and contains three identical columns of 1,2,...,nruns. In case of 
    \code{lhs.augment}, if the design to be augmented had been reordered before, the augmented design preserves this reorder 
    and also the respective numbering of the design.

    The attribute \code{design.info} is a list of various design properties, with type resolving to \dQuote{lhs}. 
    In addition to the standard list elements (cf. \code{\link[DoE.base]{design}}), the \code{subtype} 
    element indicates the type of latin hypercube designs and possibly additional augmentations, the element 
    \code{quantitative} is a vector of \code{nfactor} logical TRUEs, 
    and the \code{digits} elements indicates the digits to which the data were rounded. 
    (Note that \code{randomize} is always TRUE, \code{replications} is always 1 and \code{repeat.only} is always FALSE; 
    these elements are only present to fulfill the formal requirements for class \code{design}.)
    }
\author{ Ulrike Groemping }
\note{ This package is currently under intensive development. Substantial changes are to be expected in the near future.}
\references{ 
  Beachkofski, B., Grandhi, R. (2002) Improved Distributed Hypercube Sampling. American Institute of Aeronautics and Astronautics Paper 1274.
  
  Stein, M. (1987) Large Sample Properties of Simulations Using Latin Hypercube Sampling. Technometrics. 29, 143--151. 
  
  Stocki, R. (2005) A method to improve design reliability using optimal Latin hypercube sampling. Computer Assisted Mechanics and Engineering Sciences 12, 87--105.
}
\examples{
   plan <- lhs.design(20,7,"maximin",digits=2) 
   plan
   plot(plan)
   cor(plan)
   y <- rnorm(20)
   r.plan <- add.response(plan, y)
   plan2 <- lhs.augment(plan, m=10)
   plot(plan2)
   cor(plan2)
   plan3 <- lhs.design(20,4,"random",
          factor.names=list(c(15,25), c(10,90), c(0,120), c(12,24)), digits=2)
   plot(plan3)
   cor(plan3)
   
   plan4 <- lhs.design(20,4,"optimum",
        factor.names=list(torque=c(10,14),friction=c(25,35),
              temperature=c(-5,35),pressure=c(20,50)),digits=2)
   plot(plan4)
   cor(plan4)
}
\seealso{ See Also \code{\link[lhs]{lhs}}, \code{\link[FrF2]{FrF2}}, \code{\link[DoE.base]{oa.design}}, \code{\link[FrF2]{pb}} }
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ array }
\keyword{ design }% __ONLY ONE__ keyword per line
