% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGA.fit.R
\name{EGA.fit}
\alias{EGA.fit}
\title{\code{\link[EGAnet]{EGA}} Optimal Model Fit using the Total Entropy Fit Index  (\code{\link[EGAnet]{tefi}})}
\usage{
EGA.fit(
  data,
  n = NULL,
  corr = c("auto", "cor_auto", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  model = c("BGGM", "glasso", "TMFG"),
  algorithm = c("leiden", "louvain", "walktrap"),
  plot.EGA = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis}

\item{n}{Numeric (length = 1).
Sample size if \code{data} is a correlation matrix}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{model}{Character (length = 1).
Defaults to \code{"glasso"}.
Available options:

\itemize{

\item \code{"BGGM"} --- Computes the Bayesian Gaussian Graphical Model.
Set argument \code{ordinal.categories} to determine
levels allowed for a variable to be considered ordinal.
See \code{?BGGM::estimate} for more details

\item \code{"glasso"} --- Computes the GLASSO with EBIC model selection.
See \code{\link[EGAnet]{EBICglasso.qgraph}} for more details

\item \code{"TMFG"} --- Computes the TMFG method.
See \code{\link[EGAnet]{TMFG}} for more details

}}

\item{algorithm}{Character or \code{igraph} \code{cluster_*} function.
Three options are listed below but all are available
(see \code{\link[EGAnet]{community.detection}} for other options):

\itemize{

\item \code{"leiden"} --- See \code{\link[igraph]{cluster_leiden}} for more details.
\emph{Note}: The Leiden algorithm will default to the
Constant Potts Model objective function
(\code{objective_function = "CPM"}). Set
\code{objective_function = "modularity"} to use
modularity instead (see examples). By default, searches along
resolutions from 0 to \code{max(abs(network))} or the maximum
absolute edge weight in the network in 0.01 increments
(\code{resolution_parameter = seq.int(0, max(abs(network)), 0.01)}).
For modularity, searches along resolutions from 0 to 2 in 0.05 increments
(\code{resolution_parameter = seq.int(0, 2, 0.05)}) by default.
Use the argument \code{resolution_parameter} to change the search parameters
(see examples)

\item \code{"louvain"} --- See \code{\link[EGAnet]{community.consensus}} for more details.
By default, searches along resolutions from 0 to 2 in 0.05 increments
(\code{resolution_parameter = seq.int(0, 2, 0.05)}). Use the argument \code{resolution_parameter}
to change the search parameters (see examples)

\item \code{"walktrap"} --- This algorithm is the default. See \code{\link[igraph]{cluster_walktrap}} for more details.
By default, searches along 3 to 8 steps (\code{steps = 3:8}). Use the argument \code{steps}
to change the search parameters (see examples)

}}

\item{plot.EGA}{Boolean.
If \code{TRUE}, returns a plot of the network and its estimated dimensions.
Defaults to \code{TRUE}}

\item{verbose}{Boolean.
Whether messages and (insignificant) warnings should be output.
Defaults to \code{FALSE} (silent calls).
Set to \code{TRUE} to see all messages and warnings for every function call}

\item{...}{Additional arguments to be passed on to
\code{\link[EGAnet]{auto.correlate}}, \code{\link[EGAnet]{network.estimation}},
\code{\link[EGAnet]{community.detection}}, \code{\link[EGAnet]{community.consensus}},
and \code{\link[EGAnet]{EGA.estimate}}}
}
\value{
Returns a list containing:

\item{EGA}{\code{\link[EGAnet]{EGA}} results of the best fitting solution}

\item{EntropyFit}{\code{\link[EGAnet]{tefi}} fit values for each solution}

\item{Lowest.EntropyFit}{The best fitting solution based on \code{\link[EGAnet]{tefi}}}

\item{parameter.space}{Parameter values used in search space}
}
\description{
Estimates the best fitting model using \code{\link[EGAnet]{EGA}}.
The number of steps in the \code{\link[igraph]{cluster_walktrap}} detection
algorithm is varied and unique community solutions are compared using
\code{\link[EGAnet]{tefi}}.
}
\examples{
# Load data
wmt <- wmt2[,7:24]

# Estimate optimal EGA with Walktrap
fit.walktrap <- EGA.fit(
  data = wmt, algorithm = "walktrap",
  steps = 3:8, # default
  plot.EGA = FALSE # no plot for CRAN checks
)

# Estimate optimal EGA with Leiden and CPM
fit.leiden <- EGA.fit(
  data = wmt, algorithm = "leiden",
  objective_function = "CPM", # default
  # resolution_parameter = seq.int(0, max(abs(network)), 0.01),
  # For CPM, the default max resolution parameter
  # is set to the largest absolute edge in the network
  plot.EGA = FALSE # no plot for CRAN checks
)

# Estimate optimal EGA with Leiden and modularity
fit.leiden <- EGA.fit(
  data = wmt, algorithm = "leiden",
  objective_function = "modularity",
  resolution_parameter = seq.int(0, 2, 0.05),
  # default for modularity
  plot.EGA = FALSE # no plot for CRAN checks
)

\dontrun{
# Estimate optimal EGA with Louvain
fit.louvain <- EGA.fit(
  data = wmt, algorithm = "louvain",
  resolution_parameter = seq.int(0, 2, 0.05), # default
  plot.EGA = FALSE # no plot for CRAN checks
)}

}
\references{
\strong{Entropy fit measures} \cr
Golino, H., Moulder, R. G., Shi, D., Christensen, A. P., Garrido, L. E., Neito, M. D., Nesselroade, J., Sadana, R., Thiyagarajan, J. A., & Boker, S. M. (in press).
Entropy fit indices: New fit measures for assessing the structure and dimensionality of multiple latent variables.
\emph{Multivariate Behavioral Research}.

\strong{Simulation for EGA.fit} \cr
Jamison, L., Christensen, A. P., & Golino, H. (under review).
Optimizing Walktrap's community detection in networks using the Total Entropy Fit Index.
\emph{PsyArXiv}.

\strong{Leiden algorithm} \cr
Traag, V. A., Waltman, L., & Van Eck, N. J. (2019).
From Louvain to Leiden: guaranteeing well-connected communities.
\emph{Scientific Reports}, \emph{9}(1), 1-12.

\strong{Louvain algorithm} \cr
Blondel, V. D., Guillaume, J. L., Lambiotte, R., & Lefebvre, E. (2008).
Fast unfolding of communities in large networks.
\emph{Journal of Statistical Mechanics: Theory and Experiment}, \emph{2008}(10), P10008.

\strong{Walktrap algorithm} \cr
Pons, P., & Latapy, M. (2006).
Computing communities in large networks using random walks.
\emph{Journal of Graph Algorithms and Applications}, \emph{10}, 191-218.
}
\seealso{
\code{\link[EGAnet]{plot.EGAnet}} for plot usage in \code{EGAnet}
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
