#' Plot FAMoS Performance
#'
#' For each FAMoS run \code{famos.performance} plots the corresponding best model and information criterion.
#' @param input Either a string giving the three-digit number of the corresponding FAMoS run, e.g "004", or a matrix containing the tested models along with the respective information criteria.
#' @param path If \code{input} is the string of an FAMoS run, the directory containing the "FAMoS-Results" folder needs to be supplied as well. Default to \code{\link{getwd}}.
#' @param ic The information criterion the model selection will be based on. Options are "AICc", "AIC" and "BIC". Default to "AICc".
#' @param save.output A string containing the location and name under which the figure should be saved (format is .pdf). Default to NULL.
#' @param log If true, the results are plotted on a logarithmic scale. Default to FALSE.
#' @param plot.style Changes the style of the plot. Options are either "cross" or "block" (default).
#' @details The upper plot shows the improvement of the selected information criterion over each FAMoS iteration. The best value is shown on the right axis. The lower plot depicts the corresponding best model of each iteration. Here, green circles show added, red circles removed and blue circles swapped parameters. The parameters of the final model are printed bold.
#' @return A plot showing the value of the corresponding information criterion and best model of each FAMoS iteration.
#' @export
#' @examples
#' #plot the performance of an FAMoS run
#' famos.performance(input = famos.run, log = TRUE)


famos.performance <- function(input, path = getwd(), ic = "AICc", save.output = NULL, log = FALSE, plot.style = "block"){

  switch (ic,
          "AICc" = {ic.index <- 1},
          "AIC"  = {ic.index <- 2},
          "BIC"  = {ic.index <- 3}
  )
  if(is.character(input)){
    #read in the tested models
    if(file.exists(paste0(path,"/FAMoS-Results/TestedModels/TestedModels",input,".rds")) == FALSE){
      stop(paste0("The specified file in ", paste0(path,"/FAMoS-Results/TestedModels/TestedModels",input,".rds"), " does not exist. If the directory is incorrect, use 'path' to specify a new one."))
    }
    mt <- readRDS(paste0(path,"/FAMoS-Results/TestedModels/TestedModels",input,".rds"))
    if(is.null(mt) || ncol(mt) < 2){
      stop("famos.performance needs at least two completed runs for plotting.")
    }
  }else if(is.matrix(input)){
    mt <- input
  }else{
    stop("Input needs to be either a FAMoS run number or a matrix.")
  }
  #excluding failed runs
  if(length(which(is.na(mt[1,]))) > 0){
    mt <- mt[,-c(which(is.na(mt[1,])))]
  }
  all.names <- rownames(mt)[5:nrow(mt)]
  #create storage for individual runs and the corresponding number of jobs performed
  split.run <- list()
  jobs.per.run <- c()
  #split TestedModels into smaller chunks per model run
  for(i in unique(mt[4,])){
    jobs.per.run <- c(jobs.per.run, length(which(mt[4,] == i)))
    split.run[[i]] <- mt[,which(mt[4,] == i)]
  }

  #get the corresponding best model of each run
  get.best <- mt[,1]
  for(i in unique(mt[4,-1])){
    runs <- as.matrix(split.run[[i]])
    if(ncol(runs) == 1){
      get.best <- cbind(get.best, runs)
    }else{
      get.best <- cbind(get.best, runs[,which.min(runs[ic.index,])[1]])
    }
  }

  #replace the model if no improvement in AICC is found
  for(i in 2:ncol(get.best)){
    if(get.best[ic.index,i] > get.best[ic.index,i-1]){
      get.best[,i] <- get.best[,i-1]
      get.best[4,i] <- get.best[4,i] + 1
    }
  }

  if(is.null(save.output) == FALSE){
    grDevices::pdf(file = save.output,
                   width  = ifelse(1.5 + 0.2*max(mt[4,]) < 4, 4 ,1.5 + 0.2*max(mt[4,])),
                   height = 2.5 + 0.2*(nrow(mt)-2),
                   useDingbats = F)
  }


  #plot improvement in AICc and the corresponding models
  graphics::layout(matrix(c(1,1,2,2),ncol = 2, byrow = TRUE),
                   widths=c(3,1), heights=c(1,3))

  graphics::par(mai = c(0,
                        0.5 + 0.06*max(nchar(all.names)),
                        0.2,
                        0.2 + 0.1 * nchar(as.character(round(get.best[ic.index,ncol(get.best)],1)))))
  #plot AICc
  if(log == FALSE || (min(get.best[ic.index,]) <= 0)){
    plot.log = ""
  }else{
    plot.log = "y"
  }
  graphics::plot(unique(mt[4,]), get.best[ic.index,],
                 log = plot.log,
                 xlab = "",
                 ylab = ic,
                 type = "o",
                 lwd = 2,
                 axes = F,
                 main = "FAMoS performance")
  graphics::box()
  graphics::axis(2, las = 1, cex.axis = 0.7)
  graphics::axis(side   = 4,
                 at     = round(get.best[ic.index,ncol(get.best)],1),
                 tick   = TRUE,
                 las = 1,
                 cex.axis = 0.7)

  graphics::par(mai = c(1,
                        0.5 + 0.06 * max(nchar(all.names)),
                        0.2,
                        0.2 + 0.1 * nchar(as.character(round(get.best[ic.index,ncol(get.best)],1)))))


  if(plot.style == "cross"){
    #plot the corresponding models
    graphics::plot(0,0,
                   type = "n",
                   axes = F,
                   xlab = "iteration",
                   ylab = "",
                   xlim = c(1, max(mt[4,])),
                   ylim = c(1, nrow(mt) - 4))

    #plot the first model
    start <- which(mt[5:nrow(mt),1] == 1)
    graphics::points(rep(1, length(start)), length(all.names) + 1 - start , pch = 4)

    #plot the following best models and indicate the changes by coloured circles
    for(i in unique(mt[4,-1])){
      rowx <- which(get.best[4,] == i)
      start <- which(get.best[5:nrow(get.best),rowx] == 1)
      graphics::points(rep(i, length(start)), length(all.names) + 1 - start , pch = 4)
      diff.model <- (get.best[5:nrow(get.best),rowx] - get.best[5:nrow(get.best),rowx-1])
      if(sum(abs(diff.model)) > 0){
        if(sum(diff.model) == 0){
          crcl.col <- "blue"
        }else if(max(diff.model) == 1){
          crcl.col <- "chartreuse4"
        }else{
          crcl.col <- "red"
        }
        crcl <- which(diff.model != 0)
        graphics::points(rep(i, length(crcl)), length(all.names) + 1 - crcl , pch = 1, col = crcl.col, cex = 2, lwd = 2)
      }
    }

    graphics::box()
    graphics::axis(1, las = 1)

    for(i in 1:length(all.names)){
      graphics::axis(side   = 2,
                     at     = length(all.names) + 1 - i,
                     labels = all.names[i],
                     tick   = TRUE,
                     las = 2,
                     cex.axis = 0.7,
                     font = get.best[4 + i,ncol(get.best)] + 1)
    }
    for(i in 1:ncol(get.best)){
      graphics::abline(v = i - 0.5, col = "lightgray")
    }
    for(i in 2:(nrow(get.best) - 4)){
      graphics::abline(h = i - 0.5, col = "lightgray")
    }

  }else if(plot.style == "block"){

    image.matrix <- get.best[-c(1:4),]
    plot.matrix <- image.matrix
    for(i in 2:ncol(image.matrix)){
      image.matrix[,i] <- get.best[-c(1:4),i] - get.best[-c(1:4),i-1]
      if(is.element(-1, image.matrix[,i]) && is.element(1, image.matrix[,i])){
        plot.matrix[which(image.matrix[,i] == 1),i] <- 4
        plot.matrix[which(image.matrix[,i] == -1),i] <- 4
      }else if(is.element(-1, image.matrix[,i])){
        plot.matrix[which(image.matrix[,i] == -1),i] <- 2
      }else if(is.element(1, image.matrix[,i])){
        plot.matrix[which(image.matrix[,i] == 1),i] <- 3
      }
    }

    for(i in 1:ncol(plot.matrix)){
      plot.matrix[,i] <- rev(plot.matrix[,i])
    }

    graphics::image( 1:ncol(get.best),1:(nrow(get.best) - 4), t(plot.matrix),
                     col = c("white", "gray40","red", "chartreuse4" , "blue"),
                     breaks = c(0,0.5,1.5,2.5,3.5,4.5),
                     xlab = c("iteration"),
                     ylab = c(""),
                     axes = F)

    graphics::box()
    graphics::axis(1, las = 1)

    for(i in 1:length(all.names)){
      graphics::axis(side   = 2,
                     at     = length(all.names) + 1 - i,
                     labels = all.names[i],
                     tick   = TRUE,
                     las = 2,
                     cex.axis = 0.7,
                     font = get.best[4 + i,ncol(get.best)] + 1)
    }

    graphics::grid(nx = ncol(plot.matrix), ny = nrow(plot.matrix), lty = 1)
  }
  if(is.null(save.output) == FALSE){
    grDevices::dev.off()

  }

}
