% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discreteLR_fun.R
\name{discrete.LR}
\alias{discrete.LR}
\alias{DLR}
\alias{NDLR}
\title{Discrete Lehmann-Romano procedure}
\usage{
discrete.LR(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  direction = "sd",
  adaptive = TRUE,
  critical.values = FALSE
)

DLR(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  direction = "sd",
  critical.values = FALSE
)

NDLR(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  direction = "sd",
  critical.values = FALSE
)
}
\arguments{
\item{raw.pvalues}{vector of the raw observed p-values, as provided by the end user and before matching with their nearest neighbor in the CDFs supports.}

\item{pCDFlist}{a list of the supports of the CDFs of the p-values. Each support is represented by a vector that must be in increasing order.}

\item{alpha}{the target FDP, a number strictly between 0 and 1. For \code{*.fast} kernels, it is only necessary, if \code{stepUp = TRUE}.}

\item{zeta}{the target probability of not exceeding the desired FDP, a number strictly between 0 and 1. If \code{zeta=NULL} (the default), then \code{zeta} is chosen equal to \code{alpha}.}

\item{direction}{a character string specifying whether to conduct a step-up (\code{direction="su"}, the default) or step-down procedure (\code{direction="sd"}).}

\item{adaptive}{a boolean specifying whether to conduct an adaptive procedure or not.}

\item{critical.values}{a boolean. If \code{TRUE}, critical constants are computed and returned (this is computationally intensive).}
}
\value{
A \code{FDX} S3 class object whose elements are:
\item{Rejected}{Rejected raw p-values.}
\item{Indices}{Indices of rejected hypotheses.}
\item{Num.rejected}{Number of rejections.}
\item{Adjusted}{Adjusted p-values (only for step-down direction).}

\item{Critical.values}{Critical values (if requested).}
\item{Method}{A character string describing the used algorithm, e.g. 'Discrete Lehmann-Romano procedure (step-up)'.}
\item{FDP.threshold}{FDP threshold \code{alpha}.}
\item{Exceedance.probability}{Probability \code{zeta} of FDP exceeding \code{alpha}; thus, FDP is being controlled at level \code{alpha} with confidence \code{1 - zeta}.}


\item{Data$raw.pvalues}{The values of \code{raw.pvalues}.}
\item{Data$pCDFlist}{The values of \code{pCDFlist}.}

\item{Data$data.name}{The respective variable names of \code{raw.pvalues} and \code{pCDFlist}.}
}
\description{
Apply the [DLR] procedure, with or without computing the critical values, to
a set of p-values and their discrete support. Both step-down and step-up
procedures can be computed and non-adaptive versions are available as well.
}
\details{
\code{DLR} and \code{NDLR} are wrapper functions for \code{discrete.LR}.
The first one simply passes all its parameters to \code{discrete.LR} with
\code{adaptive = TRUE} and \code{NDLR} does the same with
\code{adaptive = FALSE}.
}
\section{References}{

S. Döhler and E. Roquain (2019). Controlling False Discovery Exceedance for
Heterogeneous Tests.
\href{https://arxiv.org/abs/1912.04607v1}{arXiv:1912.04607v1}.
}

\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

# Construction of the p-values and their supports (fisher.pvalues.support
# is from 'DiscreteFDR' package!)
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

DLR.sd.fast <- DLR(raw.pvalues, pCDFlist)
summary(DLR.sd.fast)
DLR.su.fast <- DLR(raw.pvalues, pCDFlist, direction = "su")
summary(DLR.su.fast)

DLR.sd.crit <- DLR(raw.pvalues, pCDFlist, critical.values = TRUE)
summary(DLR.sd.crit)
DLR.su.crit <- DLR(raw.pvalues, pCDFlist, direction = "su", critical.values = TRUE)
summary(DLR.su.crit)

NDLR.sd.fast <- NDLR(raw.pvalues, pCDFlist)
summary(NDLR.sd.fast)
NDLR.su.fast <- NDLR(raw.pvalues, pCDFlist, direction = "su")
summary(NDLR.su.fast)

NDLR.sd.crit <- NDLR(raw.pvalues, pCDFlist, critical.values = TRUE)
summary(NDLR.sd.crit)
NDLR.su.crit <- NDLR(raw.pvalues, pCDFlist, direction = "su", critical.values = TRUE)
summary(NDLR.su.crit)

}
\seealso{
\code{\link{kernel}}, \code{\link{FDX-package}}, \code{\link{continuous.LR}},
\code{\link{continuous.GR}}, \code{\link{discrete.GR}}, 
\code{\link{discrete.PB}}, \code{\link{weighted.LR}},
\code{\link{weighted.GR}}, \code{\link{weighted.PB}}
}
