% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fkf.R
\name{fkf}
\alias{fkf}
\title{Fast Kalman filter}
\usage{
fkf(a0, P0, dt, ct, Tt, Zt, HHt, GGt, yt)
}
\arguments{
\item{a0}{A \code{vector} giving the initial value/estimation of the state variable.}

\item{P0}{A \code{matrix} giving the variance of \code{a0}.}

\item{dt}{A \code{matrix} giving the intercept of the transition equation (see \bold{Details}).}

\item{ct}{A \code{matrix} giving the intercept of the measurement equation (see \bold{Details}).}

\item{Tt}{An \code{array} giving the factor of the transition equation (see \bold{Details}).}

\item{Zt}{An \code{array} giving the factor of the measurement equation (see \bold{Details}).}

\item{HHt}{An \code{array} giving the variance of the innovations of the transition equation (see \bold{Details}).}

\item{GGt}{An \code{array} giving the variance of the disturbances of the measurement equation (see \bold{Details}).}

\item{yt}{A \code{matrix} containing the observations. \dQuote{NA}-values are allowed (see \bold{Details}).}
}
\value{
An S3-object of class \dQuote{fkf}, which is a list with the following elements:

\tabular{rl}{
    \code{att} \tab A \eqn{m \times n}{m * n}-matrix containing the filtered state variables, i.e. att[,t] = \eqn{a_{t|t}}{a(t|t)}.\cr
    \code{at} \tab A \eqn{m \times (n + 1)}{m * (n + 1)}-matrix containing the predicted state variables, i.e. at[,t] = \eqn{a_t}{a(t)}.\cr
    \code{Ptt} \tab A \eqn{m \times m \times n}{m * m * n}-array containing the variance of \code{att}, i.e. Ptt[,,t] = \eqn{P_{t|t}}{P(t|t)}.\cr
    \code{Pt} \tab A \eqn{m \times m \times (n + 1)}{m * m * (n + 1)}-array containing the variances of \code{at}, i.e. Pt[,,t] = \eqn{P_t}{P(t)}.\cr
    \code{vt} \tab A \eqn{d \times n}{d * n}-matrix of the prediction errors i.e. vt[,t] = \eqn{v_t}{v(t)}.\cr
    \code{Ft} \tab A \eqn{d \times d \times n}{d * d * n}-array which contains the variances of \code{vt}, i.e. Ft[,,t] = \eqn{F_t}{F(t)}.\cr
    \code{Kt} \tab A \eqn{m \times d \times n}{m * d * n}-array containing the \dQuote{Kalman gain} i.e. Kt[,,t] = \eqn{k_t}{K(t)}.\cr
    \code{logLik} \tab The log-likelihood. \cr
    \code{status} \tab A vector which contains the status of LAPACK's \code{dpotri} and \code{dpotrf}. \eqn{(0, 0)} means successful exit.\cr
  \code{sys.time} \tab The time elapsed as an object of class \dQuote{proc_time}.
}
}
\description{
This function allows for fast and flexible Kalman filtering. Both, the
measurement and transition equation may be multivariate and parameters
are allowed to be time-varying. In addition \dQuote{NA}-values in the
observations are supported. \code{fkf} wraps the \code{C}-function
\code{FKF} which fully relies on linear algebra subroutines contained
in BLAS and LAPACK.
}
\details{
\strong{State space form}

The following notation is closest to the one of Koopman et al.
The state space model is represented by the transition equation and
the measurement equation. Let \eqn{m}{m} be the dimension of the state
variable, \eqn{d}{d} be the dimension of the observations, and \eqn{n}
the number of observations. The transition equation and the
measurement equation are given by
\deqn{\alpha_{t + 1} = d_t + T_t \cdot \alpha_t + H_t \cdot \eta_t}{alpha(t + 1) = d(t) + T(t) alpha(t) + H(t) * eta(t)} 
\deqn{y_t = c_t + Z_t \cdot \alpha_t + G_t \cdot \epsilon_t,}{y(t) = c(t) + Z(t) alpha(t) + G(t) * epsilon(t),}
where \eqn{\eta_t}{eta(t)} and \eqn{\epsilon_t}{epsilon(t)} are iid
\eqn{N(0, I_m)}{N(0, I(m))} and iid \eqn{N(0, I_d)}{N(0, I(d))},
respectively, and \eqn{\alpha_t}{alpha(t)} denotes the state
variable. The parameters admit the following dimensions:

\tabular{lll}{
\eqn{\alpha_{t} \in R^{m}}{alpha(t) in R^m} \tab
\eqn{d_{t} \in R^m}{d(t) in R^m} \tab
\eqn{\eta_{t} \in R^m}{eta(t) in R^m} \cr
\eqn{T_{t} \in R^{m \times m}}{T(t) in R^{m x m}} \tab
\eqn{H_{t} \in R^{m \times m}}{H(t) in R^{m x m}} \tab \cr
\eqn{y_{t} \in R^d}{y(t) in R^d} \tab
\eqn{c_t \in R^d}{c(t) in R^d} \tab
\eqn{\epsilon_{t} \in R^d}{epsilon(t) in R^d} \cr
\eqn{Z_{t} \in R^{d \times m}}{Z(t) in R^{d x m}} \tab
\eqn{G_{t} \in R^{d \times d}}{G(t) in R^{d x d}} \tab 
}

Note that \code{fkf} takes as input \code{HHt} and \code{GGt} which
corresponds to \eqn{H_t H_t^\prime}{H(t)H(t)'} and \eqn{G_t G_t^\prime}{G(t)G(t)'}. 

% <------------------------------------->
\strong{Iteration:}

The filter iterations are implemented using the expected values
\deqn{a_{t} = E[\alpha_t | y_1,\ldots,y_{t-1}]}{a(t) = E[alpha(t) | y(1),...,y(t-1)]}
\deqn{a_{t|t} = E[\alpha_t | y_1,\ldots,y_{t}]}{a(t|t) = E[alpha(t) | y(1),...,y(t)]}

and variances
\deqn{P_{t} = Var[\alpha_t | y_1,\ldots,y_{t-1}]}{P(t) = Var[alpha(t) | y(1),...,y(t-1)]}
\deqn{P_{t|t} = Var[\alpha_t | y_1,\ldots,y_{t}]}{P(t|t) = Var[alpha(t) | y(1),...,y(t)]}

of the state \eqn{\alpha_{t}}{alpha(t)} in the following way
(for the case of no NA's):

Initialisation: Set \eqn{t=1}{t=1} with \eqn{a_{t} = a0}{a(t)=a0} and \eqn{P_{t}=P0}{P(t)=P0}

Updating equations:
\deqn{v_t = y_t - c_t - Z_t a_t}{v(t) = y(t) - c(t) - Z(t) a(t)}
\deqn{F_t = Z_t P_t Z_t^{\prime} + G_t G_t^\prime}{F(t)=Z(t)P(t)Z(t)' + G(t)G(t)'}
\deqn{K_t = P_t Z_t^{\prime} F_{t}^{-1}}{K(t) = P(t) Z(t)' F(t)^{-1}}
\deqn{a_{t|t} = a_t + K_t v_t}{a(t|t) = a(t) + K(t)v(t)}
\deqn{P_{t|t} = P_t - P_t Z_t^\prime K_t^\prime}{P(t|t) = P(t) - P(t) Z(t)' K(t)'}

Prediction equations:
\deqn{a_{t+1} = d_{t} + T_{t} a_{t|t}}{a(t+1) = d(t) + T(t) a(t|t)}
\deqn{P_{t+1} = T_{t} P_{t|t} T_{t}^{\prime} + H_t H_t^\prime}{P(t+1) = T(t)P(t)T(t)' + H(t)H(t)'}

Next iteration: Set \eqn{t=t+1}{t=t+1} and goto \dQuote{Updating equations}.

% <------------------------------------->
\strong{NA-values:}

NA-values in the observation matrix \code{yt} are supported.  If
  particular observations \code{yt[,i]} contain NAs, the NA-values are
  removed and the measurement equation is adjusted accordingly.  When
  the full vector \code{yt[,i]} is missing the Kalman filter reduces to
  a prediction step.

% <------------------------------------->
\strong{Parameters:}

The parameters can either be constant or deterministic
  time-varying. Assume the number of observations is \eqn{n}
  (i.e. \eqn{y = (y_t)_{t = 1, \ldots, n}, y_t = (y_{t1}, \ldots,
  y_{td})}{y = y[,1:n]}). Then, the parameters admit the following
  classes and dimensions:

\tabular{ll}{
    \code{dt} \tab either a \eqn{m \times n}{m * n} (time-varying) or a \eqn{m \times 1}{m * 1} (constant) matrix. \cr
    \code{Tt} \tab either a \eqn{m \times m \times n}{m * m * n} or a \eqn{m \times m \times 1}{m * m * 1} array. \cr
    \code{HHt} \tab either a \eqn{m \times m \times n}{m * m * n} or a \eqn{m \times m \times 1}{m * m * 1} array. \cr
    \code{ct} \tab either a \eqn{d \times n}{d * n} or a \eqn{d \times 1}{d * 1} matrix. \cr
    \code{Zt} \tab either a \eqn{d \times m \times n}{d * m * n} or a \eqn{d \times m \times 1}{d * m * 1} array. \cr
    \code{GGt} \tab either a \eqn{d \times d \times n}{d * d * n} or a \eqn{d \times d \times 1}{d * d * 1} array. \cr
    \code{yt} \tab a \eqn{d \times n}{d * n} matrix.
  }


% <------------------------------------->
  \strong{BLAS and LAPACK routines used:}

The \R function \code{fkf} basically wraps the \code{C}-function
  \code{FKF}, which entirely relies on linear algebra subroutines
  provided by BLAS and LAPACK. The following functions are used:

\tabular{rl}{
    BLAS: \tab \code{dcopy}, \code{dgemm}, \code{daxpy}. \cr
    LAPACK: \tab \code{dpotri}, \code{dpotrf}.
  }

\code{FKF} is called through the \code{.Call} interface.  Internally,
  \code{FKF} extracts the dimensions, allocates memory, and initializes
  the \R-objects to be returned. \code{FKF} subsequently calls
  \code{cfkf} which performs the Kalman filtering.

The only critical part is to compute the inverse of \eqn{F_t}{F[,,t]}
  and the determinant of \eqn{F_t}{F[,,t]}. If the inverse can not be
  computed, the filter stops and returns the corresponding message in
  \code{status} (see \bold{Value}). If the computation of the
  determinant fails, the filter will continue, but the log-likelihood
  (element \code{logLik}) will be \dQuote{NA}.

  The inverse is computed in two steps:
First, the Cholesky factorization of \eqn{F_t}{F[,,t]} is
  calculated by \code{dpotrf}. Second, \code{dpotri} calculates the
  inverse based on the output of \code{dpotrf}. \cr
  The determinant of \eqn{F_t}{F[,,t]} is computed using again the
  Cholesky decomposition.


The first element of both \code{at} and \code{Pt} is filled with the
function arguments \code{a0} and \code{P0}, and the last, i.e. the (n +
1)-th, element of \code{at} and \code{Pt} contains the predictions for the next time step.
}
\section{References}{

  Harvey, Andrew C. (1990). \emph{Forecasting, Structural Time Series
  Models and the Kalman Filter}.  Cambridge University Press.

Hamilton, James D. (1994). \emph{Time Series Analysis}.  Princeton
University Press.

Koopman, S. J., Shephard, N., Doornik, J. A. (1999).
\emph{Statistical algorithms for models in state space using SsfPack
2.2}. Econometrics Journal, Royal Economic Society, vol. 2(1), pages
107-160.
}

\examples{
## <--------------------------------------------------------------------------->
## Example: Local level model for the Nile's annual flow.
## <--------------------------------------------------------------------------->
## Transition equation:
## alpha[t+1] = alpha[t] + eta[t], eta[t] ~ N(0, HHt)          
## Measurement equation:
## y[t] = alpha[t] + eps[t], eps[t] ~  N(0, GGt)

y <- Nile
y[c(3, 10)] <- NA  # NA values can be handled

## Set constant parameters:
dt <- ct <- matrix(0) 
Zt <- Tt <- matrix(1)
a0 <- y[1]            # Estimation of the first year flow 
P0 <- matrix(100)     # Variance of 'a0'

## Estimate parameters:
fit.fkf <- optim(c(HHt = var(y, na.rm = TRUE) * .5,
                   GGt = var(y, na.rm = TRUE) * .5),
                 fn = function(par, ...)
                 -fkf(HHt = matrix(par[1]), GGt = matrix(par[2]), ...)$logLik,
                 yt = rbind(y), a0 = a0, P0 = P0, dt = dt, ct = ct,
                 Zt = Zt, Tt = Tt)

## Filter Nile data with estimated parameters:
fkf.obj <- fkf(a0, P0, dt, ct, Tt, Zt, HHt = matrix(fit.fkf$par[1]),
               GGt = matrix(fit.fkf$par[2]), yt = rbind(y))

## Compare with the stats' structural time series implementation:
fit.stats <- StructTS(y, type = "level")

fit.fkf$par
fit.stats$coef

## Plot the flow data together with fitted local levels:
plot(y, main = "Nile flow")
lines(fitted(fit.stats), col = "green")
lines(ts(fkf.obj$att[1, ], start = start(y), frequency = frequency(y)), col = "blue")
legend("top", c("Nile flow data", "Local level (StructTS)", "Local level (fkf)"),
       col = c("black", "green", "blue"), lty = 1)

}
\seealso{
\code{\link[=plot.fkf]{plot}} to visualize and analyze \code{fkf}-objects, \code{\link{KalmanRun}} from the stats package, function \code{dlmFilter} from package \code{dlm}.
}
\keyword{algebra}
\keyword{models}
\keyword{multivariate}
