% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reco_opt.R
\name{terec}
\alias{terec}
\title{Optimal combination temporal reconciliation}
\usage{
terec(base, agg_order, comb = "ols", res = NULL, tew = "sum",
      approach = "proj", nn = NULL, settings = NULL, bounds = NULL,
      immutable = NULL, ...)
}
\arguments{
\item{base}{A (\eqn{h(k^\ast + m) \times 1}) numeric vector containing base forecasts
to be reconciled ordered from the lowest frequency to the highest frequency; \eqn{m}
is the max aggregation order, \eqn{k^\ast} is the sum of (a subset of) (\eqn{p-1})
factors of \eqn{m}, excluding \eqn{m}, and \eqn{h} is the forecast horizon for the
lowest frequency time series.}

\item{agg_order}{Highest available sampling frequency per seasonal cycle (max. order
of temporal aggregation, \eqn{m}), or a vector representing a subset of \eqn{p} factors
of \eqn{m}.}

\item{comb}{A string specifying the reconciliation method. For a complete list, see \link{tecov}.}

\item{res}{A (\eqn{N(k^\ast+m) \times 1}) optional numeric vector containing the
in-sample residuals at all the temporal frequencies ordered from the lowest frequency
to the highest frequency. This vector is used to compute come covariance matrices.}

\item{tew}{A string specifying the type of temporal aggregation. Options include:
"\code{sum}" (simple summation, \emph{default}), "\code{avg}" (average),
"\code{first}" (first value of the period), and "\code{last}"
(last value of the period).}

\item{approach}{A string specifying the approach used to compute the reconciled
forecasts. Options include:
\itemize{
\item "\code{proj}" (\emph{default}): Projection approach according to Byron (1978, 1979).
\item "\code{strc}": Structural approach as proposed by Hyndman et al. (2011).
\item "\code{proj_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for projection approach.
\item "\code{strc_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for structural approach.
}}

\item{nn}{A string specifying the algorithm to compute non-negative reconciled forecasts:
\itemize{
\item "\code{osqp}": quadratic programming optimization
(\href{https://osqp.org/}{\pkg{osqp}} solver).
\item "\code{sntz}": heuristic "set-negative-to-zero" (Di Fonzo and Girolimetto, 2023).
}}

\item{settings}{An object of class \code{osqpSettings} specifying settings
for the \href{https://osqp.org/}{\pkg{osqp}} solver. For details, refer to the
\href{https://osqp.org/}{\pkg{osqp} documentation} (Stellato et al., 2020).}

\item{bounds}{A (\eqn{(k^\ast + m) \times 2}) numeric matrix specifying the
temporal bounds. The first column represents the lower bound, and the
second column represents the upper bound.}

\item{immutable}{A matrix with two columns (\eqn{k,j}), such that
\describe{
\item{Column 1}{Denotes the temporal aggregation order (\eqn{k = m,\dots,1}).}
\item{Column 2}{Indicates the temporal forecast horizon (\eqn{j = 1,\dots,m/k}).}
}
For example, when working with a quarterly time series:
\itemize{
\item \code{t(c(4, 1))} - Fix the one step ahead annual forecast.
\item \code{t(c(1, 2))} - Fix the two step ahead quarterly forecast.
}}

\item{...}{
  Arguments passed on to \code{\link[=tecov]{tecov}}
  \describe{
    \item{\code{mse}}{If \code{TRUE} (\emph{default}) the residuals used to compute the covariance
matrix are not mean-corrected.}
    \item{\code{shrink_fun}}{Shrinkage function of the covariance matrix, \link{shrink_estim} (\emph{default})}
  }}
}
\value{
A (\eqn{h(k^\ast+m) \times 1}) numeric vector of temporal reconciled forecasts.
}
\description{
This function performs forecast reconciliation for a single time series using temporal
hierarchies (Athanasopoulos et al., 2017, Nystrup et al., 2020). The reconciled forecasts can be computed
using either a projection approach (Byron, 1978, 1979) or the equivalent structural
approach by Hyndman et al. (2011). Non-negative (Di Fonzo and Girolimetto, 2023)
and immutable reconciled forecasts can be considered.
}
\examples{
set.seed(123)
# (7 x 1) base forecasts vector (simulated), m = 4
base <- rnorm(7, rep(c(20, 10, 5), c(1, 2, 4)))
# (70 x 1) in-sample residuals vector (simulated)
res <- rnorm(70)

m <- 4 # from quarterly to annual temporal aggregation
reco <- terec(base = base, agg_order = m, comb = "wlsv", res = res)

# Immutable reconciled forecast
# E.g. fix all the quarterly forecasts
imm_q <- expand.grid(k = 1, j = 1:4)
immreco <- terec(base = base, agg_order = m, comb = "wlsv",
                 res = res, immutable = imm_q)

# Non negative reconciliation
base[7] <- -base[7] # Making negative one of the quarterly base forecasts
nnreco <- terec(base = base, agg_order = m, comb = "wlsv",
                res = res, nn = "osqp")
recoinfo(nnreco, verbose = FALSE)$info

}
\references{
Athanasopoulos, G., Hyndman, R.J., Kourentzes, N. and Petropoulos, F. (2017),
Forecasting with Temporal Hierarchies, \emph{European Journal of Operational
Research}, 262, 1, 60-74. \doi{10.1016/j.ejor.2017.02.046}

Byron, R.P. (1978), The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 141, 3, 359-367.
\doi{10.2307/2344807}

Byron, R.P. (1979), Corrigenda: The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 142(3), 405.
\doi{10.2307/2982515}

Di Fonzo, T. and Girolimetto, D. (2023), Spatio-temporal reconciliation of solar
forecasts, \emph{Solar Energy}, 251, 13–29. \doi{10.1016/j.solener.2023.01.003}

Hyndman, R.J., Ahmed, R.A., Athanasopoulos, G. and Shang, H.L. (2011),
Optimal combination forecasts for hierarchical time series,
\emph{Computational Statistics & Data Analysis}, 55, 9, 2579-2589.
\doi{10.1016/j.csda.2011.03.006}

Nystrup, P.,  \enc{Lindström}{Lindstrom}, E., Pinson, P. and Madsen, H. (2020),
Temporal hierarchies with autocorrelation for load forecasting,
\emph{European Journal of Operational Research}, 280, 1, 876-888.
\doi{10.1016/j.ejor.2019.07.061}

Stellato, B., Banjac, G., Goulart, P., Bemporad, A. and Boyd, S. (2020), OSQP:
An Operator Splitting solver for Quadratic Programs,
\emph{Mathematical Programming Computation}, 12, 4, 637-672.
\doi{10.1007/s12532-020-00179-2}
}
\seealso{
Regression-based reconciliation: 
\code{\link{csrec}()},
\code{\link{ctrec}()}

Temporal framework: 
\code{\link{teboot}()},
\code{\link{tebu}()},
\code{\link{tecov}()},
\code{\link{telcc}()},
\code{\link{temo}()},
\code{\link{tetd}()},
\code{\link{tetools}()}
}
\concept{Framework: temporal}
\concept{Reco: regression-based}
