% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_LeMans.R
\docType{methods}
\name{run_LeMans}
\alias{run_LeMans}
\alias{run_LeMans,missing-method}
\alias{run_LeMans,LeMans_param-method}
\title{Project the LeMans model}
\usage{
run_LeMans(params, ...)

\S4method{run_LeMans}{missing}(N0, Fs, tot_time, nsc, nfish, phi_min,
  mature, sc_Linf, wgt, phi, ration, other, M1, suit_M2, stored_rec_funs,
  recruit_params, eps = 1e-05)

\S4method{run_LeMans}{LeMans_param}(params, years = 10, N0 = NULL,
  effort = matrix(0, years, dim(params@Qs)[3]), Fs, intercept = 1e+10,
  slope = -5, tot_time)
}
\arguments{
\item{params}{A \linkS4class{LeMans_param} object containing the parameter values of the current LeMans model.}

\item{...}{Additional arguments.}

\item{N0}{A matrix with dimensions \code{nsc} and \code{nfish} representing the number of individuals in each length class when the model is initialised.}

\item{Fs}{An array with dimensions \code{nsc}, \code{nfish} and \code{tot_time} representing the fishing mortality of each species in each length class at each time step.}

\item{tot_time}{A numeric value representing the number of time steps to run the model for.}

\item{nsc}{A numeric value representing the number of length classes in the model.}

\item{nfish}{A numeric value representing the number of fish species in the model.}

\item{phi_min}{A numeric value representing the time step of the model.}

\item{mature}{A matrix with dimensions \code{nsc} and \code{nfish} and elements in the range 0-1 representing the proportion of individuals that are mature for each species and length class.}

\item{sc_Linf}{A numeric vector of length \code{nsc} representing the length class at which each species reaches its asymptotic length.}

\item{wgt}{A matrix with dimensions \code{nsc} and \code{nfish} representing the weight of each species in each length class.}

\item{phi}{A matrix with dimensions \code{nsc} and \code{nfish} representing the proportion of individuals that leave each length class.}

\item{ration}{A matrix with dimensions \code{nsc} and \code{nfish} representing the amount of food required for fish of a given species and length class to grow according to the von Bertalanffy growth curve in a time step.}

\item{other}{A numeric value representing the amount of other food (g) available from prey that is not explicitly represented in the model.}

\item{M1}{A matrix of dimensions \code{nsc} and \code{nfish} representing the natural mortality of each species for each length class.}

\item{suit_M2}{A list object of length \code{nfish}. Each element in the list is an array of dimensions \code{nsc}, \code{nsc} and \code{nfish} containing a value between zero and 1 representing prey preference and prey suitability for each species and length class.}

\item{stored_rec_funs}{A list object of length \code{nfish} where each element includes the stock recruitment function for each species. If an invalid recruitment function is selected, \code{NULL} is returned and a warning message is shown.}

\item{recruit_params}{A list object of length \code{nfish} specifying the parameters for the recruitment function.}

\item{eps}{A numeric value specifying a numerical offset. The default value is \code{1e-5}.}

\item{years}{A numeric value representing the number of years that the model is run for. The default is 10.}

\item{effort}{A matrix with dimensions \code{years} and the number of fishing gears, representing fishing effort in each year for each gear. This parameter is required only if \code{Fs} is missing.}

\item{intercept}{A numeric value representing the number of individuals in the first length class. This parameter is only required if \code{N0} is missing. The default is \code{1e10}.}

\item{slope}{A numeric value representing the slope of the community size spectrum. This parameter is only required if \code{N0} is missing. The default is -5.}
}
\value{
An object of class \code{\linkS4class{LeMans_outputs}}.
}
\description{
Project the LeMans model forward in time.
}
\examples{
# Run the model with all inputs specified explicitly:
# Set up the inputs to the function - species-independent parameters
nfish <- nrow(NS_par)
nsc <- 32
maxsize <- max(NS_par$Linf)*1.01 # the biggest size is 1\% bigger than the largest Linf
l_bound <- seq(0, maxsize, maxsize/nsc); l_bound <- l_bound[-length(l_bound)]
u_bound <- seq(maxsize/nsc, maxsize, maxsize/nsc)
mid <- l_bound+(u_bound-l_bound)/2

# Set up the inputs to the function - species-specific parameters
Linf <- NS_par$Linf # the von-Bertalanffy asymptotic length of each species (cm).
W_a <- NS_par$W_a # length-weight conversion parameter.
W_b <- NS_par$W_b # length-weight conversion parameter.
k <- NS_par$k # the von-Bertalnaffy growth parameter.
Lmat <- NS_par$Lmat # the length at which 50\\\% of individuals are mature (cm).

# Get phi_min
tmp <- calc_phi(k, Linf, nsc, nfish, u_bound, l_bound, calc_phi_min=FALSE,
                  phi_min=0.1) # fixed phi_min
phi <- tmp$phi
phi_min <- tmp$phi_min

# Calculate growth increments
tmp <- calc_ration_growthfac(k, Linf, nsc, nfish, l_bound, u_bound, mid, W_a, W_b, phi_min)
ration <- tmp$ration
sc_Linf <- tmp$sc_Linf
wgt <- tmp$wgt
g_eff <- tmp$g_eff

# Calculate maturity
mature <- calc_mature(Lmat, nfish, mid, kappa=rep(10, nfish), sc_Linf)

# Create recruitment functions
stored_rec_funs <- get_rec_fun(rep("hockey-stick", nfish))
recruit_params <- do.call("Map", c(c, list(a=NS_par$a, b=NS_par$b)))

# Calculate background mortality
M1 <- calc_M1(nsc, sc_Linf, phi_min)

# Calculate predator-prey size preferences
prefs <- calc_prefs(pred_mu=-2.25, pred_sigma=0.5, wgt, sc_Linf)

# Calculate prey preference and prey suitability
suit_M2 <- calc_suit_vect(nsc, nfish, sc_Linf, prefs, NS_tau)

# Calculate catchability
Qs <- calc_Q(curve=rep("logistic", nfish), species=NS_par$species_names,
             max_catchability=rep(1, nfish), gear_name=NS_par$species_names,
             nsc=nsc, nfish=nfish, mid=mid, l_bound=l_bound, u_bound=u_bound,
             species_names=NS_par$species_names, eta=rep(0.25, nfish), L50=Lmat)

# Get an initial population
N0 <- get_N0(nsc, nfish, mid, wgt, sc_Linf, intercept=1e10, slope=-5)
years <- 10 # run the model for 10 years
tot_time <- years*phi_min # total number of time steps

# Define fishing effort to be 0.5 for all species
effort <- matrix(0.5, tot_time, dim(Qs)[3])

# Calculate F
Fs <- array(0, dim=c(nsc, nfish, tot_time))
for (j in 1:ncol(effort)) {
  for (ts in 1:tot_time) {
    Fs[,,ts] <- Fs[,,ts]+effort[ts, j]*Qs[,,j]
  }
}

# Run the model
model_run <- run_LeMans(N0=N0, tot_time=tot_time, Fs=Fs, nsc=nsc, nfish=nfish,
                        phi_min=phi_min, mature=mature, sc_Linf=sc_Linf, wgt=wgt,
                        phi=phi, ration=ration, other=NS_other, M1=M1, suit_M2=suit_M2,
                        stored_rec_funs=stored_rec_funs, recruit_params=recruit_params,
                        eps=1e-05)

##############################################
# Alternatively:
NS_params <- LeMansParam(NS_par,tau=NS_tau,eta=rep(0.25,21),L50=NS_par$Lmat,other=NS_other)

# Define fishing effort
effort <- matrix(0.5, 10, dim(NS_params@Qs)[3])

# Run the model
model_run <- run_LeMans(NS_params, years=10, effort=effort)
}
\seealso{
\code{\linkS4class{LeMans_outputs}}, \code{\linkS4class{LeMans_param}}, \code{\link{LeMansParam}}
}
