% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeItemsScale.R
\name{makeItemsScale}
\alias{makeItemsScale}
\title{scale items from a summated scale}
\usage{
makeItemsScale(scale, lowerbound, upperbound, items, variance = 0.5)
}
\arguments{
\item{scale}{(int) a vector or dataframe of the summated rating scale.
Should range from ('lowerbound' * 'items') to ('upperbound' * 'items')}

\item{lowerbound}{(int) lower bound of the scale item
(example: '1' in a '1' to '5' rating)}

\item{upperbound}{(int) upper bound of the scale item
(example: '5' in a '1' to '5' rating)}

\item{items}{(positive, int) k, or number of columns to generate}

\item{variance}{(positive, real) standard deviation of values sampled
from a normally-distributed log transformation. Default = '0.5'.

A value of '0' makes all values in the correlation matrix the same,
equal to the mean correlation needed to produce the desired \emph{Cronbach's Alpha}.
A value of '2', or more, risks producing a matrix that is not positive-
definite, so not feasible.}
}
\value{
a dataframe with 'items' columns and 'length(scale)' rows
}
\description{
\code{makeItemsScale()} generates a random dataframe
of scale items based on a predefined summated scale,
such as created by the \code{lfast()} function.

scale, lowerbound, upperbound, items
}
\examples{

## define parameters
items <- 4
lowerbound <- 1
upperbound <- 5

## scale properties
n <- 64
mean <- 3.5
sd <- 1.00

## create scale
set.seed(42)
meanScale <- lfast(
  n = n, mean = mean, sd = sd,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items
)
summatedScale <- meanScale * items

## create items
newItems <- makeItemsScale(
  scale = summatedScale,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items
)
str(newItems)

##
## Testing Lowest value to Highest value of a scale
##
lowerbound <- 1
upperbound <- 5
items <- 6

# lowest to highest values
myvalues <- c((lowerbound * items):(upperbound * items))

## Low variance usually gives higher Cronbach's Alpha
mydat_20 <- makeItemsScale(
  scale = myvalues,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items, variance = 0.20
)

str(mydat_20)

moments <- data.frame(
  means = apply(mydat_20, MARGIN = 2, FUN = mean) |> round(3),
  sds = apply(mydat_20, MARGIN = 2, FUN = sd) |> round(3)
) |> t()

moments

cor(mydat_20) |> round(2)
alpha(mydat_20) |> round(2)


## default variance
mydat_50 <- makeItemsScale(
  scale = myvalues,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items, variance = 0.50
)

str(mydat_50)

moments <- data.frame(
  means = apply(mydat_50, MARGIN = 2, FUN = mean) |> round(3),
  sds = apply(mydat_50, MARGIN = 2, FUN = sd) |> round(3)
) |> t()

moments

cor(mydat_50) |> round(2)
alpha(mydat_50) |> round(2)


## higher variance usually gives lower Cronbach's Alpha
mydat_80 <- makeItemsScale(
  scale = myvalues,
  lowerbound = lowerbound, upperbound = upperbound,
  items = items, variance = 0.80
)

str(mydat_80)

moments <- data.frame(
  means = apply(mydat_80, MARGIN = 2, FUN = mean) |> round(3),
  sds = apply(mydat_80, MARGIN = 2, FUN = sd) |> round(3)
) |> t()

moments

cor(mydat_80) |> round(2)
alpha(mydat_80) |> round(2)

}
