% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_FiniteMixture.R
\name{calc_FiniteMixture}
\alias{calc_FiniteMixture}
\title{Apply the finite mixture model (FMM) after Galbraith (2005) to a given De
distribution}
\usage{
calc_FiniteMixture(
  data,
  sigmab,
  n.components,
  grain.probability = FALSE,
  dose.scale,
  pdf.weight = TRUE,
  pdf.sigma = "sigmab",
  pdf.colors = "gray",
  pdf.scale,
  plot.proportions = TRUE,
  plot = TRUE,
  ...
)
}
\arguments{
\item{data}{\linkS4class{RLum.Results} or \link{data.frame} (\strong{required}):
for \link{data.frame}: two columns with De \code{(data[,1])} and De error \code{(values[,2])}}

\item{sigmab}{\link{numeric} (\strong{required}):
spread in De values given as a fraction (e.g. 0.2). This value represents the expected
overdispersion in the data should the sample be well-bleached
(Cunningham & Wallinga 2012, p. 100).}

\item{n.components}{\link{numeric} (\strong{required}):
number of components to be fitted. If a vector is provided (e.g. \code{c(2:8)}) the
finite mixtures for 2, 3 ... 8 components are calculated and a plot and a
statistical evaluation of the model performance (BIC score and maximum
log-likelihood) is provided.}

\item{grain.probability}{\link{logical} (\emph{with default}):
prints the estimated probabilities of which component each grain is in}

\item{dose.scale}{\link{numeric}:
manually set the scaling of the y-axis of the first plot with a vector
in the form of \code{c(min, max)}}

\item{pdf.weight}{\link{logical} (\emph{with default}):
weight the probability density functions by the components proportion (applies only
when a vector is provided for \code{n.components})}

\item{pdf.sigma}{\link{character} (\emph{with default}):
if \code{"sigmab"} the components normal distributions are plotted with a common standard
deviation (i.e. \code{sigmab}) as assumed by the FFM. Alternatively,
\code{"se"} takes the standard error of each component for the sigma
parameter of the normal distribution}

\item{pdf.colors}{\link{character} (\emph{with default}):
colour coding of the components in the the plot.
Possible options are \code{"gray"}, \code{"colors"} and \code{"none"}}

\item{pdf.scale}{\link{numeric}:
manually set the max density value for proper scaling of the x-axis of the first plot}

\item{plot.proportions}{\link{logical} (\emph{with default}):
plot \link[graphics:barplot]{graphics::barplot} showing the proportions of components if
\code{n.components} a vector with a length > 1 (e.g., \code{n.components = c(2:3)})}

\item{plot}{\link{logical} (\emph{with default}):  plot output}

\item{...}{further arguments to pass. See details for their usage.}
}
\value{
Returns a plot (\emph{optional}) and terminal output. In addition an
\linkS4class{RLum.Results} object is returned containing the
following elements:

\item{.$summary}{\link{data.frame} summary of all relevant model results.}
\item{.$data}{\link{data.frame} original input data}
\item{.$args}{\link{list} used arguments}
\item{.$call}{\link{call} the function call}
\item{.$mle}{ covariance matrices of the log likelihoods}
\item{.$BIC}{ BIC score}
\item{.$llik}{ maximum log likelihood}
\item{.$grain.probability}{ probabilities of a grain belonging to a component}
\item{.$components}{\link{matrix} estimates of the de, de error and proportion for each component}
\item{.$single.comp}{\link{data.frame} single component FFM estimate}

If a vector for \code{n.components} is provided (e.g.  \code{c(2:8)}),
\code{mle} and \code{grain.probability} are lists containing matrices of the
results for each iteration of the model.

The output should be accessed using the function \link{get_RLum}
}
\description{
This function fits a k-component mixture to a De distribution with differing
known standard errors. Parameters (doses and mixing proportions) are
estimated by maximum likelihood assuming that the log dose estimates are
from a mixture of normal distributions.
}
\details{
This model uses the maximum likelihood and Bayesian Information Criterion
(BIC) approaches.

Indications of overfitting are:
\itemize{
\item increasing BIC
\item repeated dose estimates
\item covariance matrix not positive definite
\item covariance matrix produces \code{NaN}
\item convergence problems
}

\strong{Plot}

If a vector (\code{c(k.min:k.max)}) is provided
for \code{n.components} a plot is generated showing the the k components
equivalent doses as normal distributions. By default \code{pdf.weight} is
set to \code{FALSE}, so that the area under each normal distribution is
always 1. If \code{TRUE}, the probability density functions are weighted by
the components proportion for each iteration of k components, so the sum of
areas of each component equals 1. While the density values are on the same
scale when no weights are used, the y-axis are individually scaled if the
probability density are weighted by the components proportion.\cr
The standard deviation (sigma) of the normal distributions is by default
determined by a common \code{sigmab} (see \code{pdf.sigma}). For
\code{pdf.sigma = "se"} the standard error of each component is taken
instead.\cr
The stacked \link[graphics:barplot]{graphics::barplot} shows the proportion of each component (in
per cent) calculated by the FFM. The last plot shows the achieved BIC scores
and maximum log-likelihood estimates for each iteration of k.
}
\section{Function version}{
 0.4.2
}

\examples{

## load example data
data(ExampleData.DeValues, envir = environment())

## (1) apply the finite mixture model
## NOTE: the data set is not suitable for the finite mixture model,
## which is why a very small sigmab is necessary
calc_FiniteMixture(ExampleData.DeValues$CA1,
                   sigmab = 0.2, n.components = 2,
                   grain.probability = TRUE)

## (2) repeat the finite mixture model for 2, 3 and 4 maximum number of fitted
## components and save results
## NOTE: The following example is computationally intensive. Please un-comment
## the following lines to make the example work.
FMM<- calc_FiniteMixture(ExampleData.DeValues$CA1,
                         sigmab = 0.2, n.components = c(2:4),
                         pdf.weight = TRUE, dose.scale = c(0, 100))

## show structure of the results
FMM

## show the results on equivalent dose, standard error and proportion of
## fitted components
get_RLum(object = FMM, data.object = "components")

} 

\section{How to cite}{
Burow, C., 2022. calc_FiniteMixture(): Apply the finite mixture model (FMM) after Galbraith (2005) to a given De distribution. Function version 0.4.2. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., 2022. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.9.18. https://CRAN.R-project.org/package=Luminescence
}

\references{
Galbraith, R.F. & Green, P.F., 1990. Estimating the component
ages in a finite mixture. Nuclear Tracks and Radiation Measurements 17,
197-206.

Galbraith, R.F. & Laslett, G.M., 1993. Statistical models
for mixed fission track ages. Nuclear Tracks Radiation Measurements 4,
459-470.

Galbraith, R.F. & Roberts, R.G., 2012. Statistical aspects of
equivalent dose and error calculation and display in OSL dating: An overview
and some recommendations. Quaternary Geochronology 11, 1-27.

Roberts, R.G., Galbraith, R.F., Yoshida, H., Laslett, G.M. & Olley, J.M., 2000.
Distinguishing dose populations in sediment mixtures: a test of single-grain
optical dating procedures using mixtures of laboratory-dosed quartz.
Radiation Measurements 32, 459-465.

Galbraith, R.F., 2005. Statistics for Fission Track Analysis, Chapman & Hall/CRC, Boca Raton.

\strong{Further reading}

Arnold, L.J. & Roberts, R.G., 2009. Stochastic
modelling of multi-grain equivalent dose (De) distributions: Implications
for OSL dating of sediment mixtures. Quaternary Geochronology 4,
204-230.

Cunningham, A.C. & Wallinga, J., 2012. Realizing the
potential of fluvial archives using robust OSL chronologies. Quaternary
Geochronology 12, 98-106.

Rodnight, H., Duller, G.A.T., Wintle, A.G. &
Tooth, S., 2006. Assessing the reproducibility and accuracy of optical
dating of fluvial deposits.  Quaternary Geochronology 1, 109-120.

Rodnight, H. 2008. How many equivalent dose values are needed to obtain a
reproducible distribution?. Ancient TL 26, 3-10.
}
\seealso{
\link{calc_CentralDose}, \link{calc_CommonDose},
\link{calc_FuchsLang2001}, \link{calc_MinDose}
}
\author{
Christoph Burow, University of Cologne (Germany) \cr
Based on a rewritten S script of Rex Galbraith, 2006.
, RLum Developer Team}
