#' "Penalty" constraint handling method for MOEA/D
#'
#' Uses the Penalty Function constraint handling method to generate a
#' preference index for the MOEADr framework.
#'
#' This function calculates the preference index of a set of neighborhoods
#' based on the "penalty" constraint handling method. Please
#' see [order_neighborhood()] for more information on the preference index
#' matrix.
#'
#' @param B Matrix of neighborhoods (generated by [define_neighborhood()]`$B`)
#' @param bigZ Matrix of scalarized objective values for each neighborhood and
#' the incumbent solution (generated by [scalarize_values()])
#' @param bigV Matrix of violation values for each neighborhood and the
#' incumbent solution (generated in [order_neighborhood()])
#' @param beta Penalization constant (non-negative value)
#' @param ... other parameters (unused, included for compatibility with
#' generic call)
#'
#' @return `[ N x (T+1) ]` matrix of preference indices. Each row `i` contains
#' a permutation of `{1, 2, ..., (T+1)}`, where `1,...,T` correspond
#' to the solutions contained in the neighborhood of the i-th subproblem,
#' `B[i, ]`, and `T+1` corresponds to the incumbent solution for that
#' subproblem. The order of the permutation is defined by the increasing values
#' of `f(xk) + beta * v(xk)`, where `f(xk)` is the aggregation function value of
#' the k-th solution being compared, and v(xk) is its total constraint violation
#' (calculated in [evaluate_population()]`$V$v`).
#'
#' @export

constraint_penalty <- function(B, bigZ, bigV, beta, ...)
{
  # ========== Error catching and default value definitions
  assertthat::assert_that(
    identical(dim(bigZ), dim(bigV)),
    is.numeric(beta),
    beta >= 0)
  # ==========

  # Calculate penalized values
  bigZV <- bigZ + beta * bigV

  # Get the selection matrix for all neighborhoods
  sel.indx <- t(apply(bigZV,
                      MARGIN = 2,
                      FUN    = order))

  return(sel.indx)
}
