% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{MoE_estep}
\alias{MoE_estep}
\title{Compute the Responsility Matrix and Log-likelihood for MoEClust Mixture Models}
\usage{
MoE_estep(modelName, data, mus, sigs, log.tau = 0, Vinv = NULL,
  Dens = NULL)
}
\arguments{
\item{modelName}{A character string indicating the model. The help file for \code{\link[mclust]{mclustModelNames}} describes the available models.}

\item{data}{If there are no expert network covariates, \code{data} should be a numeric matrix or data frame, wherein rows correspond to observations (n) and columns correspond to variables (d). If there are expert network covariates, this should be a list of length G containing matrices/data.frames of (multivariate) WLS residuals for each component.}

\item{mus}{The mean for each of G components. If there is more than one component, this is a matrix whose k-th column is the mean of the k-th component of the mixture model. For the univariate models, this is a G-vector of means. In the presence of expert network covariates, all values should be equal to zero.}

\item{sigs}{A list of length G of variance parameters of the model. The components of this list depend on the specification of \code{modelName}.}

\item{log.tau}{If covariates enter the gating network, an n times G matrix of mixing proportions, otherwise a G-vector of mixing proportions for the components of the mixture. \strong{Must} be on the log-scale in both cases. The default of \code{0} effectively means densities (or log-densities) aren't scaled by the mixing proportions.}

\item{Vinv}{An estimate of the reciprocal hypervolume of the data region. The default is determined by applying the function \code{\link[mclust]{hypvol}} to the data. Used only if an initial guess as to which observations are noise is supplied. Mixing proportion(s) must be included for the noise component also.}

\item{Dens}{(Optional) A numeric matrix whose \code{[i,k]}-th entry is the \strong{log}-density of observation \emph{i} in component \emph{k}, scaled by the mixing proportions, to which the softmax function is to be applied, typically obtained by \code{\link{MoE_dens}} but this is not necessary. If this is supplied, all other arguments are ignored, otherwise \code{\link{MoE_dens}} is called according to the other supplied arguments.}
}
\value{
A list containing two elements:
\item{\code{z}}{A matrix with n rows and G columns containing the probability of cluster membership for each of n observations and G clusters}
\item{\code{loglik}}{The log-likelihood, computed efficiently via \code{\link[matrixStats]{rowLogSumExps}}}
}
\description{
Softmax function to compute the responsibility matrix z and the log-likelihood for MoEClust models, with the aid of \code{\link{MoE_dens}}.
}
\note{
This softmax function is intended for joint use with \code{\link{MoE_dens}}, using the \strong{log}-densities.
}
\examples{
data(ais)
hema   <- ais[,3:7]
model  <- MoE_clust(hema, G=3, gating= ~ BMI + sex, model="EEE", network.data=ais)
Dens   <- MoE_dens(modelName=model$modelName, data=hema,
                   mus=model$parameters$mean, sigs=model$parameters$variance$sigma,
                   log.tau=log(model$parameters$pro))

# Construct the z matrix and compute the log-likelihood
Estep  <- MoE_estep(Dens=Dens)
(ll    <- Estep$loglik)

# The z matrix will be close but not exactly the same as that from the model
# as the EM algorithm finishes on an M-step, but the classification should be
identical(max.col(Estep$z), as.integer(unname(model$classification))) #TRUE
round(sum(Estep$z - model$z), options()$digits) == 0                  #TRUE

# Call MoE_estep directly
Estep2 <- MoE_estep(modelName=model$modelName, data=hema,
                    mus=model$parameters$mean, sigs=model$parameters$variance$sigma,
                    log.tau=log(model$parameters$pro))
identical(Estep2$loglik, ll)                                          #TRUE
}
\seealso{
\code{\link{MoE_dens}}, \code{\link{MoE_clust}}, \code{\link[matrixStats]{rowLogSumExps}}, \code{\link[mclust]{mclustModelNames}}
}
\author{
Keefe Murphy - <\email{keefe.murphy@ucd.ie}>
}
\keyword{clustering}
