% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MR.quantreg.r
\name{MR.quantreg}
\alias{MR.quantreg}
\title{Multiply Robust Estimation of the Quantile Regression Coefficients}
\usage{
MR.quantreg(quantreg.model, imp.model = NULL, mis.model = NULL,
  L = 10, data, bootstrap = FALSE, bootstrap.size = 500,
  alpha = 0.05)
}
\arguments{
\item{quantreg.model}{The quantile regression model of interest, defined by \code{\link{def.quantreg}}.}

\item{imp.model}{A list of imputation models defined by \code{\link{def.glm}}. Within the list, each element is a list of imputation models for the missing variables. One and only one model is allowed to be specified for each missing variable. Separate multiple working models for the same variable into different lists.}

\item{mis.model}{A list of missingness probability models defined by \code{\link{def.glm}}. The dependent variable is always specified as \code{R}.}

\item{L}{Number of random draws from the estimated imputation model.}

\item{data}{A data frame with missing data encoded as \code{NA}.}

\item{bootstrap}{Logical. Should a bootstrap method be applied to calculate the standard error of the estimator and construct a Wald confidence interval for the quantile regression coefficients. Default is \code{FALSE}.}

\item{bootstrap.size}{A numeric value. Number of bootstrap resamples generated if \code{bootstrap} is \code{TRUE}. Default is 500.}

\item{alpha}{Significance level used to construct the 100(1 - alpha)\% Wald confidence interval.}
}
\value{
\item{\code{Estimate}}{A table containing the estimated quantile regression coefficients. If \code{bootstrap} is \code{TRUE}, bootstrap standard errors \code{SE} of the estimates and Wald confidence intervals are also included.}
\item{\code{fit}}{A fitted object of class inheriting from \code{\link[quantreg]{rq}} on \code{quantreg.model}.}
}
\description{
\code{MR.quantreg()} is used to estimate the quantile regression coefficients. Both missing response and/or missing covariates are allowed. Multiple missingness probability models and imputation models are allowed.
}
\examples{
# Missing covariate and missing response simultaneously
# Simulated data set
set.seed(123)
n <- 400
gamma0 <- c(1, 2, 3)
alpha0 <- c(-0.8, -0.5, 0.3)
S <- runif(n, min = -2.5, max = 2.5) # auxiliary information
S.2 <- S ^ 2
X1 <- rbinom(n, size = 1, prob = 0.5) # covariate X1
X2 <- rexp(n) # covariate X2
p.obs <- 1 / (1 + exp(alpha0[1] + alpha0[2] * S + alpha0[3] * S.2)) # missingness probability
R <- rbinom(n, size = 1, prob = p.obs)
a.x <- gamma0[1] + gamma0[2] * X1 + gamma0[3] * X2
Y <- rnorm(n, a.x)
dat <- data.frame(S, X1, X2, Y)
dat[R == 0, c(2, 4)] <- NA # X1 and Y are missing

# Estimating quantile regression coefficients
# quantile regression model of interest
reg <- def.quantreg(formula = Y ~ X1 + X2)
# imputation models for X1
impX1.1 <- def.glm(formula = X1 ~ S, family = binomial(link = logit))
impX1.2 <- def.glm(formula = X1 ~ S + X2, family = binomial(link = cloglog))
# imputation models for Y
impY.1 <- def.glm(formula = Y ~ S, family = gaussian)
impY.2 <- def.glm(formula = Y ~ S + X2, family = gaussian)
# missingness probability models
mis1 <- def.glm(formula = R ~ S + S.2, family = binomial(link = logit))
mis2 <- def.glm(formula = R ~ S.2, family = binomial(link = cloglog))

imp1 <- list(impX1.1, impY.1) # 1st set of imputation models for X1 and Y
imp2 <- list(impX1.2, impY.2) # 2nd set of imputation models for X1 and Y
imp3 <- list(impX1.1, impY.2) # 3rd set of imputation models for X1 and Y
imp4 <- list(impX1.2, impY.1) # 4th set of imputation models for X1 and Y

results <- MR.quantreg(quantreg.model = reg, imp.model = list(imp1, imp2, imp3, imp4),
                       mis.model = list(mis1, mis2), L = 5, data = dat)
}
\references{
Han, P., Kong, L., Zhao, J. and Zhou, X. (2018). A general framework for quantile estimation with incomplete data. \emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)}, in press.
}
