% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obn_bb.R
\name{ob_numerical_bb}
\alias{ob_numerical_bb}
\title{Optimal Binning for Numerical Variables using Branch and Bound Algorithm}
\usage{
ob_numerical_bb(
  feature,
  target,
  min_bins = 3,
  max_bins = 5,
  bin_cutoff = 0.05,
  max_n_prebins = 20,
  is_monotonic = TRUE,
  convergence_threshold = 1e-06,
  max_iterations = 1000
)
}
\arguments{
\item{feature}{A numeric vector representing the continuous predictor variable
to be binned. NA values are handled by exclusion during the pre-binning phase.}

\item{target}{An integer vector of binary outcomes (0/1) corresponding to
each observation in \code{feature}. Must have the same length as \code{feature}.}

\item{min_bins}{Integer. The minimum number of bins to produce. Must be \eqn{\ge} 2.
Defaults to 3.}

\item{max_bins}{Integer. The maximum number of bins to produce. Must be \eqn{\ge}
\code{min_bins}. Defaults to 5.}

\item{bin_cutoff}{Numeric. The minimum fraction of total observations required
for a bin to be considered valid. Bins with frequency < \code{bin_cutoff}
will be merged with neighbors. Value must be in (0, 1). Defaults to 0.05.}

\item{max_n_prebins}{Integer. The number of initial quantiles to generate
during the pre-binning phase. Higher values provide more granular starting
points but increase computation time. Must be \eqn{\ge} \code{min_bins}.
Defaults to 20.}

\item{is_monotonic}{Logical. If \code{TRUE}, the algorithm enforces a strict
monotonic relationship (increasing or decreasing) between the bin indices
and their WoE values. This makes the variable more interpretable for linear
models. Defaults to \code{TRUE}.}

\item{convergence_threshold}{Numeric. The threshold for the change in total
IV to determine convergence during the iterative merging process.
Defaults to 1e-6.}

\item{max_iterations}{Integer. Safety limit for the maximum number of merging
iterations. Defaults to 1000.}
}
\value{
A list containing the binning results:
  \itemize{
    \item \code{id}: Integer vector of bin identifiers (1 to k).
    \item \code{bin}: Character vector of bin labels in interval notation
          (e.g., \code{"(0.5;1.2]"}).
    \item \code{woe}: Numeric vector of Weight of Evidence for each bin.
    \item \code{iv}: Numeric vector of Information Value contribution per bin.
    \item \code{count}: Integer vector of total observations per bin.
    \item \code{count_pos}: Integer vector of positive cases (target=1) per bin.
    \item \code{count_neg}: Integer vector of negative cases (target=0) per bin.
    \item \code{cutpoints}: Numeric vector of upper boundaries for the bins
          (excluding Inf).
    \item \code{converged}: Logical indicating if the algorithm converged properly.
    \item \code{iterations}: Integer count of iterations performed.
    \item \code{total_iv}: The total Information Value of the binned variable.
  }
}
\description{
Performs supervised discretization of continuous numerical variables using a
Branch and Bound-style approach. This algorithm optimally creates bins based
on the relationship with a binary target variable, maximizing Information
Value (IV) while optionally enforcing monotonicity in Weight of Evidence (WoE).
}
\details{
The algorithm proceeds in several distinct phases to ensure stability and
optimality:

\enumerate{
  \item \strong{Pre-binning:} The numerical feature is initially discretized
  into \code{max_n_prebins} using quantiles. This handles outliers and
  provides a granular starting point.

  \item \strong{Rare Bin Management:} Bins containing fewer observations
  than the threshold defined by \code{bin_cutoff} are iteratively merged
  with their nearest neighbors to ensure statistical robustness.

  \item \strong{Monotonicity Enforcement (Optional):} If \code{is_monotonic = TRUE},
  the algorithm checks if the WoE trend is strictly increasing or decreasing.
  If not, it simulates merges in both directions to find the path that
  preserves the maximum possible Information Value while satisfying the
  monotonicity constraint.

  \item \strong{Optimization Phase:} The algorithm iteratively merges adjacent
  bins that have the lowest contribution to the total Information Value (IV).
  This process continues until the number of bins is reduced to \code{max_bins}
  or the change in IV falls below \code{convergence_threshold}.
}

\strong{Information Value (IV) Interpretation:}
\itemize{
  \item \eqn{< 0.02}: Not predictive
  \item \eqn{0.02 \text{ to } 0.1}: Weak predictive power
  \item \eqn{0.1 \text{ to } 0.3}: Medium predictive power
  \item \eqn{0.3 \text{ to } 0.5}: Strong predictive power
  \item \eqn{> 0.5}: Suspiciously high (check for leakage)
}
}
\examples{
# Example: Binning a variable with a sigmoid relationship to target
set.seed(123)
n <- 1000
# Generate feature
feature <- rnorm(n)

# Generate target based on logistic probability
prob <- 1 / (1 + exp(-2 * feature))
target <- rbinom(n, 1, prob)

# Perform Optimal Binning
result <- ob_numerical_bb(feature, target,
  min_bins = 3,
  max_bins = 5,
  is_monotonic = TRUE
)

# Check results
print(data.frame(
  Bin = result$bin,
  Count = result$count,
  WoE = round(result$woe, 4),
  IV = round(result$iv, 4)
))

cat("Total IV:", result$total_iv, "\n")

}
