% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/log_prior_pdf.R
\name{log_prior_pdf}
\alias{log_prior_pdf}
\title{Compute the natural log of the PDF for the parameters in an E-family design}
\usage{
log_prior_pdf(alpha, bpopdescr, ddescr, return_gradient = F,
  return_hessian = F)
}
\arguments{
\item{alpha}{A parameter vector.}

\item{bpopdescr}{Matrix defining the fixed effects, per row (row number =
parameter_number) we should have: \itemize{ \item column 1 the type of the
distribution for E-family designs (0 = Fixed, 1 = Normal, 2 = Uniform, 3 =
User Defined Distribution, 4 = lognormal and 5 = truncated normal) \item
column 2  defines the mean. \item column 3 defines the variance of the
distribution (or length of uniform distribution). }}

\item{ddescr}{Matrix defining the diagnonals of the IIV (same logic as for
the \code{bpopdescr}).}

\item{return_gradient}{Should the gradient be returned.}

\item{return_hessian}{Should the hessian be returned?}
}
\description{
Compute the natural log of the PDF for the parameters in an E-family design
}
\examples{
library(PopED)

############# START #################
## Create PopED database
## (warfarin model for optimization)
#####################################

## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

## Optimization using an additive + proportional reidual error  
## to avoid sample times at very low concentrations (time 0 or very late samples).

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.sd.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
  return(parameters) 
}

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.add.prop",
                                  bpop=c(CL=0.15, V=8, KA=1.0, Favail=1), 
                                  notfixed_bpop=c(1,1,1,0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=c(0.01,0.25),
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0.01,
                                  maxxt=120,
                                  a=70,
                                  mina=0.01,
                                  maxa=100)

############# END ###################
## Create PopED database
## (warfarin model for optimization)
#####################################


# Adding 40\% Uncertainty to fixed effects log-normal (not Favail)
bpop_vals <- c(CL=0.15, V=8, KA=1.0, Favail=1)
bpop_vals_ed_ln <- cbind(ones(length(bpop_vals),1)*4, # log-normal distribution
                         bpop_vals,
                         ones(length(bpop_vals),1)*(bpop_vals*0.4)^2) # 40\% of bpop value
bpop_vals_ed_ln["Favail",]  <- c(0,1,0)
bpop_vals_ed_ln

# then compute the log density 
alpha <- bpop_vals_ed_ln[bpop_vals_ed_ln[,1]!=0,2]
log_prior_pdf(alpha=alpha, bpopdescr=bpop_vals_ed_ln, ddescr=poped.db$parameters$d)


# Adding 10\% Uncertainty to fixed effects normal-distribution (not Favail)
bpop_vals_ed_n <- cbind(ones(length(bpop_vals),1)*1, # log-normal distribution
                        bpop_vals,
                        ones(length(bpop_vals),1)*(bpop_vals*0.1)^2) # 10\% of bpop value
bpop_vals_ed_n["Favail",]  <- c(0,1,0)
bpop_vals_ed_n

# then compute the log density from log_prior_pdf
alpha <- bpop_vals_ed_n[bpop_vals_ed_n[,1]!=0,2]
log_prior_pdf(alpha=alpha, bpopdescr=bpop_vals_ed_n, ddescr=poped.db$parameters$d)


# Adding 10\% Uncertainty to fixed effects uniform-distribution (not Favail)
bpop_vals_ed_u <- cbind(ones(length(bpop_vals),1)*2, # uniform distribution
                        bpop_vals,
                        ones(length(bpop_vals),1)*(bpop_vals*0.1)) # 10\% of bpop value
bpop_vals_ed_u["Favail",]  <- c(0,1,0)
bpop_vals_ed_u

# then compute the log density 
alpha <- bpop_vals_ed_ln[bpop_vals_ed_u[,1]!=0,2]
log_prior_pdf(alpha=alpha, bpopdescr=bpop_vals_ed_u, ddescr=poped.db$parameters$d)

}
\keyword{internal}

