% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate.fim.R
\name{evaluate.fim}
\alias{evaluate.fim}
\title{Evaluate the Fisher Information Matrix (FIM)}
\usage{
evaluate.fim(
  poped.db,
  fim.calc.type = NULL,
  approx.method = NULL,
  FOCE.num = NULL,
  bpop.val = NULL,
  d_full = NULL,
  docc_full = NULL,
  sigma_full = NULL,
  model_switch = NULL,
  ni = NULL,
  xt = NULL,
  x = NULL,
  a = NULL,
  groupsize = NULL,
  deriv.type = NULL,
  ...
)
}
\arguments{
\item{poped.db}{A PopED database.}

\item{fim.calc.type}{The method used for calculating the FIM. Potential values:
\itemize{
\item 0 = Full FIM.  No assumption that fixed and random effects are uncorrelated.  
\item 1 = Reduced FIM. Assume that there is no correlation in the FIM between the fixed and random effects, and set these elements in 
the FIM to zero. 
\item 2 = weighted models (placeholder).
\item 3 = Not currently used.
\item 4 = Reduced FIM and computing all derivatives with respect to the standard deviation of the residual unexplained variation (sqrt(SIGMA) in NONMEM). 
This matches what is done in PFIM, and assumes that the standard deviation of the residual unexplained variation is the estimated parameter
(NOTE: NONMEM estimates the variance of the residual unexplained variation by default). 
\item 5 = Full FIM parameterized with A,B,C matrices & derivative of variance. 
\item 6 = Calculate one model switch at a time, good for large matrices. 
\item 7 = Reduced FIM parameterized with A,B,C matrices & derivative of variance.
}}

\item{approx.method}{Approximation method for model, 0=FO, 1=FOCE, 2=FOCEI, 3=FOI}

\item{FOCE.num}{Number individuals in each step of FOCE approximation method}

\item{bpop.val}{The fixed effects parameter values.  Supplied as a vector.}

\item{d_full}{A between subject variability matrix (OMEGA in NONMEM).}

\item{docc_full}{A between occasion variability matrix.}

\item{sigma_full}{A residual unexplained variability matrix (SIGMA in NONMEM).}

\item{model_switch}{A matrix that is the same size as xt, specifying which model each sample belongs to.}

\item{ni}{A vector of the number of samples in each group.}

\item{xt}{A matrix of sample times.  Each row is a vector of sample times for a group.}

\item{x}{A matrix for the discrete design variables.  Each row is a group.}

\item{a}{A matrix of covariates.  Each row is a group.}

\item{groupsize}{A vector of the number of individuals in each group.}

\item{deriv.type}{A number indicating the type of derivative to use:
\itemize{
\item 0=Complex difference 
\item 1=Central difference 
\item 20=Analytic derivative (placeholder) 
\item 30=Automatic differentiation (placeholder)
}}

\item{...}{Other arguments passed to the function.}
}
\value{
The FIM.
}
\description{
Compute the FIM given the model, parameters, design and methods defined in the 
PopED database. Some of the arguments coming from the PopED database can be overwritten;  
by default these arguments are \code{NULL} in the 
function, if they are supplied then they are used instead of the arguments from the PopED database.
}
\examples{
## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

library(PopED)

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.md.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
    return(parameters) 
}

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.prop",
                                  bpop=c(CL=0.15, V=8, KA=1.0, Favail=1), 
                                  # notfixed_bpop=c(1,1,1,0),
                                  notfixed_bpop=c(CL=1,V=1,KA=1,Favail=0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=0.01,
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0,
                                  maxxt=120,
                                  a=70)


## evaluate initial design with the reduced FIM
FIM.1 <- evaluate.fim(poped.db) 
FIM.1
det(FIM.1)
get_rse(FIM.1,poped.db)

## evaluate initial design with the full FIM
FIM.0 <- evaluate.fim(poped.db,fim.calc.type=0) 
FIM.0
det(FIM.0)
get_rse(FIM.0,poped.db,fim.calc.type=0)

## evaluate initial design with the reduced FIM 
## computing all derivatives with respect to the 
## standard deviation of the residual unexplained variation 
FIM.4 <- evaluate.fim(poped.db,fim.calc.type=4) 
FIM.4
det(FIM.4)
get_rse(FIM.4,poped.db,fim.calc.type=4)

## evaluate initial design with the full FIM with A,B,C matricies
## should give same answer as fim.calc.type=0
FIM.5 <- evaluate.fim(poped.db,fim.calc.type=5) 
FIM.5
det(FIM.5)
get_rse(FIM.5,poped.db,fim.calc.type=5)

## evaluate initial design with the reduced FIM with 
## A,B,C matricies and derivative of variance
## should give same answer as fim.calc.type=1 (default)
FIM.7 <- evaluate.fim(poped.db,fim.calc.type=7) 
FIM.7
det(FIM.7)
get_rse(FIM.7,poped.db,fim.calc.type=7)

## evaluate FIM and rse with prior FIM.1
poped.db.prior = create.poped.database(poped.db, prior_fim = FIM.1)
FIM.1.prior <- evaluate.fim(poped.db.prior)
all.equal(FIM.1.prior,FIM.1) # the FIM is only computed from the design in the poped.db
get_rse(FIM.1.prior,poped.db.prior) # the RSE is computed with the prior information
}
\seealso{
Other FIM: 
\code{\link{LinMatrixH}()},
\code{\link{LinMatrixLH}()},
\code{\link{LinMatrixL_occ}()},
\code{\link{calc_ofv_and_fim}()},
\code{\link{ed_laplace_ofv}()},
\code{\link{ed_mftot}()},
\code{\link{efficiency}()},
\code{\link{evaluate.e.ofv.fim}()},
\code{\link{gradf_eps}()},
\code{\link{mf3}()},
\code{\link{mf7}()},
\code{\link{mftot}()},
\code{\link{ofv_criterion}()},
\code{\link{ofv_fim}()}

Other evaluate_design: 
\code{\link{evaluate_design}()},
\code{\link{evaluate_power}()},
\code{\link{get_rse}()},
\code{\link{model_prediction}()},
\code{\link{plot_efficiency_of_windows}()},
\code{\link{plot_model_prediction}()}

Other evaluate_FIM: 
\code{\link{calc_ofv_and_fim}()},
\code{\link{evaluate.e.ofv.fim}()},
\code{\link{ofv_fim}()}
}
\concept{FIM}
\concept{evaluate_FIM}
\concept{evaluate_design}
