% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.data.R
\name{sim.data}
\alias{sim.data}
\title{Generate Response}
\usage{
sim.data(
  Q = NULL,
  N = NULL,
  IQ = list(P0 = NULL, P1 = NULL),
  model = "GDINA",
  distribute = "uniform",
  control = NULL,
  verbose = TRUE
)
}
\arguments{
\item{Q}{The Q-matrix. A random 30 × 5 Q-matrix (\code{\link[Qval]{sim.Q}}) will be used if \code{Q = NULL}.}

\item{N}{Sample size. Default = 500.}

\item{IQ}{A list containing two \eqn{I}-length vectors: \code{P0} and \code{P1}.
\code{P0} represents the probability of examinees who have not mastered any attributes  
(\eqn{[00...0]}) correctly answering the item, while \code{P1} represents the probability  
of examinees who have mastered all attributes (\eqn{[11...1]}) correctly answering the item.}

\item{model}{Type of model to be fitted; can be \code{"GDINA"}, \code{"LCDM"}, \code{"DINA"}, \code{"DINO"},
\code{"ACDM"}, \code{"LLM"}, or \code{"rRUM"}.}

\item{distribute}{Attribute distributions; can be \code{"uniform"} for the uniform distribution,
\code{"mvnorm"} for the multivariate normal distribution (Chiu, Douglas, & Li,
2009) and \code{"horder"} for the higher-order distribution (Tu et al., 2022).}

\item{control}{A list of control parameters with elements:
\itemize{
    \item \code{sigma}  A positive-definite symmetric matrix specifying the variance-covariance
                       matrix when \code{distribute = "mvnorm"}. Default = 0.5 (Chiu, Douglas, & Li, 2009).
    \item \code{cutoffs}  A vector giving the cutoff for each attribute when \code{distribute = "mvnorm"}.
                         Default = \eqn{k/(1+K)} (Chiu, Douglas, & Li, 2009).
    \item \code{theta} A vector of length N representing the higher-order ability for each examinee.
                      By default, generate randomly from the standard normal distribution (Tu et al, 2022).
    \item \code{a} The slopes for the higher-order model when \code{distribute = "horder"}.
                  Default = 1.5 (Tu et al, 2022).
    \item \code{b} The intercepts when \code{distribute = "horder"}. By default, select equally spaced
                  values between -1.5 and 1.5 according to the number of attributes (Tu et al, 2022).
 }}

\item{verbose}{Logical indicating to print information or not. Default is \code{TRUE}}
}
\value{
Object of class \code{sim.data}.
An \code{sim.data} object initially gained by \code{\link[GDINA]{simGDINA}} function form \code{GDINA} package.
Elements that can be extracted using method extract include:
\describe{
 \item{dat}{An \code{N} × \code{I} simulated item response matrix.}
 \item{Q}{The Q-matrix.}
 \item{attribute}{An \code{N} × \code{K} matrix for inviduals' attribute patterns.}
 \item{catprob.parm}{A list of non-zero success probabilities for each attribute mastery pattern.}
 \item{delta.parm}{A list of delta parameters.}
 \item{higher.order.parm}{Higher-order parameters.}
 \item{mvnorm.parm}{Multivariate normal distribution parameters.}
 \item{LCprob.parm}{A matrix of success probabilities for each attribute mastery pattern.}
}
}
\description{
randomly generate response matrix according to certain conditions,
including attributes distribution, item quality, sample size, Q-matrix and cognitive diagnosis models (CDMs).
}
\examples{

################################################################
#                           Example 1                          #
#          generate data follow the uniform distrbution        #
################################################################
library(Qval)

set.seed(123)

K <- 5
I <- 10
Q <- sim.Q(K, I)

IQ <- list(
  P0 = runif(I, 0.0, 0.2),
  P1 = runif(I, 0.8, 1.0)
)

data.obj <- sim.data(Q = Q, N = 100, IQ=IQ, model = "GDINA", distribute = "uniform")

print(data.obj$dat)

################################################################
#                           Example 2                          #
#          generate data follow the mvnorm distrbution         #
################################################################
set.seed(123)
K <- 5
I <- 10
Q <- sim.Q(K, I)

IQ <- list(
  P0 = runif(I, 0.0, 0.2),
  P1 = runif(I, 0.8, 1.0)
)

cutoffs <- sample(qnorm(c(1:K)/(K+1)), ncol(Q))
data.obj <- sim.data(Q = Q, N = 10, IQ=IQ, model = "GDINA", distribute = "mvnorm",
                 control = list(sigma = 0.5, cutoffs = cutoffs))

print(data.obj$dat)

#################################################################
#                            Example 3                          #
#           generate data follow the horder distrbution         #
#################################################################
set.seed(123)
K <- 5
I <- 10
Q <- sim.Q(K, I)

IQ <- list(
  P0 = runif(I, 0.0, 0.2),
  P1 = runif(I, 0.8, 1.0)
)

theta <- rnorm(10, 0, 1)
b <- seq(-1.5,1.5,length.out=K)
data.obj <- sim.data(Q = Q, N = 10, IQ=IQ, model = "GDINA", distribute = "horder",
                 control = list(theta = theta, a = 1.5, b = b))

print(data.obj$dat)

}
\references{
Chiu, C.-Y., Douglas, J. A., & Li, X. (2009). Cluster Analysis for Cognitive Diagnosis: Theory and Applications. Psychometrika, 74(4), 633-665. DOI: 10.1007/s11336-009-9125-0.

Tu, D., Chiu, J., Ma, W., Wang, D., Cai, Y., & Ouyang, X. (2022). A multiple logistic regression-based (MLR-B) Q-matrix validation method for cognitive diagnosis models:A confirmatory approach. Behavior Research Methods. DOI: 10.3758/s13428-022-01880-x.
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
