#' Represents fields of \code{RDML} object as \code{data.frame}
#' 
#' Represents specified fields of \code{RDML} object as \code{data.frame} to 
#' overview, filter or pass them to \code{GetFData} and \code{SetFData} 
#' functions.
#' 
#' By default input this function forms \code{data.frame} with such columns: 
#' \describe{ \item{exp.id}{experiment$id} \item{run.id}{run$id} 
#' \item{react.id}{react$id} 
#' \item{position}{react$position} 
#' \item{sample}{react$sample}
#' \item{target}{data$id} \item{target.dyeId}{target[[data$id]]$dyeId} 
#' \item{sample.type}{sample[[react$sample]]$type} } You can overload default 
#' columns list by parameter \code{.default} but note that columns \code{exp.id,
#' run.id, react.id, target} are necessary for usage \code{AsTable} output as 
#' input for \code{GetFData} and \code{SetFData}. \cr Additional columns can be 
#' introduced by specifying them at input parameter \code{...} (see Examples). 
#' All default and additional columns accession expressions have to be 
#' named!!!\cr
#' 
#' Experiment, run, react and data to which belongs each fluorescence data vector
#' can be accessed by \code{experiment, run, react, data} (see Examples).
#' 
#' @param .default \code{list} of default columns
#' @param name.pattern expression to form \code{fdata.name} (see Examples)
#' @param ... additional columns
#' @author Konstantin A. Blagodatskikh <k.blag@@yandex.ru>, Stefan Roediger 
#'   <stefan.roediger@@b-tu.de>, Michal Burdukiewicz 
#'   <michalburdukiewicz@@gmail.com>
#' @keywords manip
#' @docType methods
#' @name RDML.AsTable
#' @rdname astable-method
#' @include RDML.R
#' @examples
#' \dontrun{
#' ## internal dataset stepone_std.rdml (in 'data' directory)
#' ## generated by Applied Biosystems Step-One. Contains qPCR data.
#' library(chipPCR)
#' PATH <- path.package("RDML")
#' filename <- paste(PATH, "/extdata/", "stepone_std.rdml", sep ="")
#' stepone <- RDML$new(filename)
#' ## Mark fluorescense data which Cq > 30 and add quantities to
#' ## AsTable output. 
#' ## Names for fluorescense data will contain sample name and react 
#' ## positions
#' tab <- stepone$AsTable(
#'          name.pattern = paste(react$sample$id, react$position),
#'          cq30 = if(data$cq >= 30) ">=30" else "<30",
#'          quantity = as.factor(sample[[react$sample$id]]$quantity$value)
#'          )
#' ## Show cq30 and quantities
#' tab[c("cq30", "quantity")]
#' ## Get fluorescence values for 'std' type samples 
#' ## in format ready for ggplot function
#' library(dplyr)
#' fdata <- stepone$GetFData(
#'            filter(tab, sample.type == "std"),
#'            long.table = TRUE)
#' ## Plot fdata with colour by cq30 and shape by quantity
#' library(ggplot2)
#' ggplot(fdata, aes(x = cyc, y = fluor,
#'                   group = fdata.name,
#'                   colour = cq30,
#'                   shape = quantity)) +
#'                   geom_line() + geom_point()
#' }
RDML$set("public", "AsTable",
         function(
           .default = list(
             exp.id = experiment$id$id,
             run.id = run$id$id,
             react.id = react$id$id,
             position = react$Position(run$pcrFormat),
             sample = react$sample$id,
             target = data$tar$id,
             target.dyeId = target[[data$tar$id]]$dyeId$id,
             sample.type = sample[[react$sample$id]]$type$value,
             adp = !is.null(data$adp),
             mdp = !is.null(data$mdp)),
           name.pattern = paste(
             react$Position(run$pcrFormat),
             react$sample$id,
             private$.sample[[react$sample$id]]$type$value,
             data$tar$id,
             sep = "_"),                    
           ...) {
           # create short names
           dateMade <- private$.dateMade
           dateUpdated <- private$.dateUpdated
           id <- private$.id
           experimenter <- private$.experimenter
           documentation <- private$.documentation
           dye <- private$.dye
           sample <- private$.sample
           target <- private$.target
           thermalCyclingConditions <- private$.thermalCyclingConditions
           # dilutions <- private$.dilutions
           # conditions <- private$.conditions
           
           out<-data.frame()
           for(experiment in private$.experiment) {                      
             for(run in experiment$run) {                                    
               for(react in run$react) {                          
                 for(data in react$data){
                   out<-rbind(out,
                              data.frame(
                                eval(substitute(list(
                                  .default,
                                  ...))),
                                row.names = eval(substitute(name.pattern)),
                                stringsAsFactors = FALSE
                              ))
                 }
               }
             }
           }
           cbind(fdata.name = rownames(out), out, stringsAsFactors = FALSE)
         }, 
         overwrite = TRUE)