% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mpm_to_.R
\name{age_from_stage}
\alias{age_from_stage}
\alias{mpm_to_mx}
\alias{mpm_to_lx}
\alias{mpm_to_px}
\alias{mpm_to_hx}
\title{Calculate age-specific traits from a matrix population model}
\usage{
mpm_to_mx(matU, matR, start = 1L, xmax = 1e+05, lx_crit = 1e-04, tol = 1e-04)

mpm_to_lx(matU, start = 1L, xmax = 1e+05, lx_crit = 1e-04, tol = 1e-04)

mpm_to_px(matU, start = 1L, xmax = 1e+05, lx_crit = 1e-04, tol = 1e-04)

mpm_to_hx(matU, start = 1L, xmax = 1e+05, lx_crit = 1e-04, tol = 1e-04)
}
\arguments{
\item{matU}{The survival component of a MPM (i.e. a
square projection matrix reflecting survival-related transitions; e.g.
progression, stasis, and retrogression). Optionally with named rows and
columns indicating the corresponding life stage names.}

\item{matR}{The reproductive component of a MPM (i.e. a
square projection matrix reflecting transitions due to reproduction; either
sexual, clonal, or both). Optionally with named rows and columns indicating
the corresponding life stage names.}

\item{start}{The index (or stage name) of the first stage at which the author
considers the beginning of life. Defaults to 1. Alternately, a numeric vector
giving the starting population vector (in which case \code{length(start)}
must match \code{ncol(matU))}. See section \emph{Starting from multiple stages}.}

\item{xmax}{Maximum age to which age-specific traits will be calculated
(defaults to \code{100000}).}

\item{lx_crit}{Minimum value of lx to which age-specific traits will be
calculated (defaults to \code{0.0001}).}

\item{tol}{To account for floating point errors that occasionally lead to
values of lx slightly greater than 1, values of lx within the open interval
(\code{1}, \code{1 + tol}) are coerced to 1. Defaults to \code{0.0001}. To
prevent coercion, set \code{tol} to \code{0}.}
}
\value{
A vector
}
\description{
These functions use age-from-stage decomposition methods to calculate
age-specific survivorship (lx), survival probability (px), mortality hazard
(hx), or reproduction (mx) from a matrix population model (MPM). A detailed
description of these methods can be found in sections 5.3.1 and 5.3.2 of
Caswell (2001).
}
\note{
The output vector is calculated recursively until the age class (x)
  reaches \code{xmax} or survivorship (lx) falls below \code{lx_crit},
  whichever comes first. To force calculation to \code{xmax}, set
  \code{lx_crit} to \code{0}. Conversely, to force calculation to
  \code{lx_crit}, set \code{xmax} to \code{Inf}.
}
\section{Starting from multiple stages}{

Rather than specifying argument \code{start} as a single stage class from
which all individuals start life, it may sometimes be desirable to allow for
multiple starting stage classes. For example, if users want to start their
calculation of age-specific traits from reproductive maturity (i.e. first
reproduction), they should account for the possibility that there may be
multiple stage classes in which an individual could first reproduce.

To specify multiple starting stage classes, users should specify argument
\code{start} as the desired starting population vector (\strong{n1}), giving
the proportion of individuals starting in each stage class (the length of
\code{start} should match the number of columns in the relevant MPM).

See function \code{\link{mature_distrib}} for calculating the proportion of
individuals achieving reproductive maturity in each stage class.
}

\examples{
data(mpm1)

# age-specific survivorship
mpm_to_lx(mpm1$matU)
mpm_to_lx(mpm1$matU, start = 2)       # starting from stage 2
mpm_to_lx(mpm1$matU, start = "small") # equivalent using named life stages
mpm_to_lx(mpm1$matU, xmax = 10)       # to a maximum age of 10
mpm_to_lx(mpm1$matU, lx_crit = 0.05)  # to a minimum lx of 0.05

# age-specific survival probability
mpm_to_px(mpm1$matU)

# age-specific mortality hazard
mpm_to_hx(mpm1$matU)

# age-specific fecundity
mpm_to_mx(mpm1$matU, mpm1$matF)


### starting from first reproduction
repstages <- repro_stages(mpm1$matF)
n1 <- mature_distrib(mpm1$matU, start = 2, repro_stages = repstages)

mpm_to_lx(mpm1$matU, start = n1)
mpm_to_px(mpm1$matU, start = n1)
mpm_to_hx(mpm1$matU, start = n1)
mpm_to_mx(mpm1$matU, mpm1$matF, start = n1)

}
\references{
Caswell, H. 2001. Matrix Population Models: Construction,
  Analysis, and Interpretation. Sinauer Associates; 2nd edition. ISBN:
  978-0878930968
}
\seealso{
\code{\link{lifetable_convert}}

Other life tables: 
\code{\link{lifetable_convert}},
\code{\link{mpm_to_table}()},
\code{\link{qsd_converge}()}
}
\author{
Roberto Salguero-Gómez <rob.salguero@zoo.ox.ac.uk>

Hal Caswell <h.caswell@uva.nl>

Owen R. Jones <jones@biology.sdu.dk>

Patrick Barks <patrick.barks@gmail.com>
}
\concept{life tables}
