% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EstSAEMspatial_USER.R
\name{SAEM.sclm}
\alias{SAEM.sclm}
\title{ML estimation of spatial censored linear models via the SAEM algorithm}
\usage{
SAEM.sclm(y, x, ci, lcl = NULL, ucl = NULL, coords, phi0, nugget0,
  type = "exponential", kappa = NULL, lower = c(0.01, 0.01),
  upper = c(30, 30), MaxIter = 300, M = 20, pc = 0.2, error = 1e-04,
  show_se = TRUE)
}
\arguments{
\item{y}{vector of responses of length \eqn{n}.}

\item{x}{design matrix of dimensions \eqn{n\times q}, where \eqn{q} is the number
of fixed effects, including the intercept.}

\item{ci}{vector of censoring indicators of length \eqn{n}. For each observation:
\code{1} if censored/missing, \code{0} otherwise.}

\item{lcl, ucl}{vectors of length \eqn{n} representing the lower and upper bounds
of the interval, which contains the true value of the censored observation. Default
\code{=NULL}, indicating no-censored data. For each observation: \code{lcl=-Inf} and
\code{ucl=c} (left censoring); \code{lcl=c} and \code{ucl=Inf} (right censoring); and
\code{lcl} and \code{ucl} must be finite for interval censoring. Moreover, missing data
could be defined by setting \code{lcl=-Inf} and \code{ucl=Inf}.}

\item{coords}{2D spatial coordinates of dimensions \eqn{n\times 2}.}

\item{phi0}{initial value for the spatial scaling parameter.}

\item{nugget0}{initial value for the nugget effect parameter.}

\item{type}{type of spatial correlation function: \code{'exponential'},
\code{'gaussian'}, \code{'matern'}, and \code{'pow.exp'} for exponential, gaussian,
matérn, and power exponential, respectively.}

\item{kappa}{parameter for some spatial correlation functions. See \code{\link{CovMat}}.}

\item{lower, upper}{vectors of lower and upper bounds for the optimization method.
If unspecified, the default is \code{c(0.01,0.01)} for lower and \code{c(30,30)} for upper.}

\item{MaxIter}{maximum number of iterations of the SAEM algorithm. By default \code{=300}.}

\item{M}{number of Monte Carlo samples for stochastic approximation. By default \code{=20}.}

\item{pc}{percentage of initial iterations of the SAEM algorithm with no memory.
It is recommended that \code{50<MaxIter*pc<100}. By default \code{=0.20}.}

\item{error}{maximum convergence error. By default \code{=1e-4}.}

\item{show_se}{logical. It indicates if the standard errors
should be estimated by default \code{=TRUE}.}
}
\value{
An object of class "sclm". Generic functions \code{print} and \code{summary} have
methods to show the results of the fit. The function \code{plot} can extract
convergence graphs for the parameter estimates.

Specifically, the following components are returned:
\item{Theta}{estimated parameters in all iterations, \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{theta}{final estimation of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{beta}{estimated \eqn{\beta}.}
\item{sigma2}{estimated \eqn{\sigma^2}.}
\item{phi}{estimated \eqn{\phi}.}
\item{tau2}{estimated \eqn{\tau^2}.}
\item{EY}{stochastic approximation of the first conditional moment.}
\item{EYY}{stochastic approximation of the second conditional moment.}
\item{SE}{vector of standard errors of \eqn{\theta = (\beta, \sigma^2, \phi, \tau^2)}.}
\item{InfMat}{observed information matrix.}
\item{loglik}{log-likelihood for the SAEM method.}
\item{AIC}{Akaike information criterion.}
\item{BIC}{Bayesian information criterion.}
\item{Iter}{number of iterations needed to converge.}
\item{time}{processing time.}
\item{call}{\code{RcppCensSpatial} call that produced the object.}
\item{tab}{table of estimates.}
\item{critFin}{selection criteria.}
\item{range}{effective range.}
\item{ncens}{number of censored/missing observations.}
\item{MaxIter}{maximum number of iterations for the SAEM algorithm.}
}
\description{
It fits the left, right, or interval spatial censored linear model using the
Stochastic Approximation EM (SAEM) algorithm. It provides estimates and standard
errors of the parameters and supports missing values on the dependent variable.
}
\details{
The spatial Gaussian model is given by

\eqn{Y = X\beta + \xi},

where \eqn{Y} is the \eqn{n\times 1} response vector, \eqn{X} is the \eqn{n\times q}
design matrix, \eqn{\beta} is the \eqn{q\times 1} vector of regression coefficients
to be estimated, and \eqn{\xi} is the error term which is normally distributed with
zero-mean and covariance matrix \eqn{\Sigma=\sigma^2 R(\phi) + \tau^2 I_n}. We assume
that \eqn{\Sigma} is non-singular and \eqn{X} has full rank \insertCite{diggle2007springer}{RcppCensSpatial}.

The estimation process is performed via the SAEM algorithm, initially proposed by
\insertCite{delyon1999convergence;textual}{RcppCensSpatial}. The spatial censored
(SAEM) algorithm was previously proposed by \insertCite{lachos2017influence;textual}{RcppCensSpatial} and
\insertCite{ordonez2018geostatistical;textual}{RcppCensSpatial} and is available in the package \code{CensSpatial}.
These packages differ in the random number generation and optimization procedure.

This model is also a particular case of the spatio-temporal model defined by
\insertCite{valeriano2021likelihood;textual}{RcppCensSpatial} when the number of
temporal observations is equal to one. The computing codes of the spatio-temporal
SAEM algorithm are available in the package \code{StempCens}.
}
\note{
The SAEM final estimates correspond to the estimates obtained at the last iteration
of the algorithm.

To fit a regression model for non-censored data, just set \code{ci} as a vector of zeros.
}
\examples{
\donttest{# Example 1: 8\% of right-censored observations
set.seed(1000)
n = 50   # Test with another values for n
coords = round(matrix(runif(2*n,0,15),n,2), 5)
x = cbind(rnorm(n), rnorm(n))
data = rCensSp(c(4,-2), 1, 3, 0.50, x, coords, "right", 0.08)

fit = SAEM.sclm(y=data$y, x=x, ci=data$ci, lcl=data$lcl, ucl=data$ucl,
                coords, phi0=2, nugget0=1, type="exponential", M=10,
                pc=0.18)
fit

# Example 2: censored and missing observations
set.seed(123)
n = 200
coords = round(matrix(runif(2*n,0,20),n,2), 5)
x = cbind(runif(n), rnorm(n), rexp(n))
data = rCensSp(c(1,4,-1), 2, 3, 0.50, x, coords, "left", 0.05, 0,
               "matern", 3)
data$y[c(10,120)] = NA
data$ci[c(10,120)] = 1
data$ucl[c(10,120)] = Inf

fit2 = SAEM.sclm(y=data$y, x=x, ci=data$ci, lcl=data$lcl, ucl=data$ucl,
                 coords, phi0=2, nugget0=1, type="matern", kappa=3,
                 M=10, pc=0.18)
fit2$tab
plot(fit2)}
}
\references{
\insertAllCited
}
\seealso{
\code{\link{EM.sclm}}, \code{\link{MCEM.sclm}}, \code{\link{predict.sclm}}
}
\author{
Katherine L. Valeriano, Alejandro Ordoñez, Christian E. Galarza, and Larissa A. Matos.
}
