% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oneStep.like.R
\name{oneStep.like}
\alias{oneStep.like}
\title{Mixture of two uniforms likelihood}
\usage{
oneStep.like(a, dist, covars, w.hi = NULL)
}
\arguments{
\item{a}{A vector or matrix of covariate 
and expansion term 
coefficients. If matrix, dimension is 
\emph{k} X \emph{p}, where 
\emph{k} = \code{nrow(a)}) is the number of coefficient
vectors to evaluate (cases) and \emph{p} 
= \code{ncol(a)})
is the number of covariate and expansion 
coefficients in the likelihood (i.e., rows are 
cases and columns are covariates). If \code{a} is a 
dimensionless vector, it is interpreted as a 
single row with \emph{k} = 1. 
Covariate coefficients in \code{a} are the first 
\emph{q} values (\emph{q} <= \emph{p}), and must 
be on a log scale.}

\item{dist}{A numeric vector of length \emph{n} or 
a single-column matrix (dimension \emph{n}X1) containing 
detection distances at which to evaluate the likelihood.}

\item{covars}{A numeric vector of length \emph{q} or a
matrix of dimension \emph{n}X\emph{q} containing 
covariate values 
associated with distances in argument \code{dist}.}

\item{w.hi}{A numeric scalar containing maximum 
distance. The right-hand cutoff or upper limit. 
Ignored by some likelihoods (such as halfnorm, 
negexp, and hazrate), but is a fixed parameter
in other likelihoods (such as oneStep, heber, 
and uniform).}
}
\value{
A list containing the following two components:
\itemize{
  \item \bold{L.unscaled}: A matrix of size 
   \emph{n}X\emph{k}X\emph{b} 
   containing likelihood values evaluated at 
   distances in \code{dist}.
   Each row is associated with 
   a single distance, and each column is associated with 
   a single case (row of \code{a}).  This matrix is  
   "unscaled" because the underlying likelihood does 
   not integrate to one. Values in \code{L.unscaled} 
   are always greater than or equal to zero.
   
 \item \bold{params}: A \emph{n}X\emph{k}X\emph{b} array 
 of the 
 likelihood's (canonical) parameters. First page contains 
 parameter values related to covariates (i.e., 
 \emph{s = exp(x'a)}),
 while subsequent pages contain other parameters. 
 \emph{b} = 1 for halfnorm, negexp; \emph{b} = 2 for 
 hazrate and others.
 Rows correspond to distances in \code{dist}. Columns 
 correspond to rows from argument \code{a}. 
}
}
\description{
Compute likelihood function for a mixture of two uniform
distributions.
}
\details{
Rdistance's \code{oneStep} likelihood is a mixture of two 
non-overlapping uniform distributions. The 'oneStep' density function
is  
\deqn{f(d|p, \theta) = \frac{p}{\theta}I(0 \leq d \leq \theta) + 
       \frac{1 - p}{w - \theta}I(\theta \le d \leq w),}{
       f(d|p,T) = (p/T)I(0<=d<=T) + ((1-p)/(w-T))I(T<d<=w),}
where \eqn{I(x)} is the indicator function for event \eqn{x}, 
and \eqn{w} is the nominal strip width (i.e., \code{w.hi - w.lo} in Rdistance). 
The unknown parameters to be estimated 
are \eqn{\theta}{T} and \eqn{p}{p}
(\eqn{w} is fixed - given by the user). 
 
Covariates influence values of \eqn{\theta}{T} 
via a log link function, i.e., \eqn{\theta = e^{x'b}}{T = exp(x'b)},
where \eqn{x} is the vector of covariate values 
associated with distance \eqn{d}, and \eqn{b}
is the vector of estimated coefficients.
}
\examples{

# Fit oneStep to simulated data
whi <- 250
T <- 100  # true threshold
p <- 0.85
n <- 200 
x <- c( runif(n*p, min=0, max=T), runif(n*(1-p), min=T, max=whi))
x <- setUnits(x, "m")
tranID <- sample(rep(1:10, each=n/10), replace=FALSE)
detectDf <- data.frame(transect = tranID, dist = x)
siteDf <- data.frame(transect = 1:10
               , length = rep(setUnits(10,"m"), 10))
distDf <- RdistDf(siteDf, detectDf)

# Estimation
fit <- dfuncEstim(distDf
                 , formula = dist ~ 1
                 , likelihood = "oneStep"
                 , w.hi = setUnits(whi, "m")
                 )
plot(fit)
thetaHat <- exp(coef(fit)[1]) 
pHat <- coef(fit)[2]
c(thetaHat, pHat) # should be close to c(100,0.85)

summary(abundEstim(fit, ci=NULL)) 

}
\references{
Peter F. Craigmile & D.M. Tirrerington (1997) "Parameter estimation for 
finite mixtures of uniform distributions", 
Communications in Statistics - Theory and Methods, 26:8, 1981-1995, 
DOI: 10.1080/03610929708832026

 A. Hussein & J. Liu (2009) "Parametric estimation of mixtures of two 
 uniform distributions", Journal of Statistical Computation and Simulation, 
 79:4, 395-410, DOI:10.1080/00949650701810406
}
\seealso{
\code{\link{dfuncEstim}}, 
         \code{\link{abundEstim}},
         other \code{<likelihood>.like} functions
}
