% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curve_distlp.R
\name{riem.distlp}
\alias{riem.distlp}
\title{Distance between Two Curves on Manifolds}
\usage{
riem.distlp(
  riemobj1,
  riemobj2,
  vect = NULL,
  geometry = c("intrinsic", "extrinsic"),
  ...
)
}
\arguments{
\item{riemobj1}{a S3 \code{"riemdata"} class for \eqn{N} manifold-valued data along the curve.}

\item{riemobj2}{a S3 \code{"riemdata"} class for \eqn{N} manifold-valued data along the curve.}

\item{vect}{a vector of domain values. If given \code{Null} (default), sequence \code{1:N} is set.}

\item{geometry}{(case-insensitive) name of geometry; either geodesic (\code{"intrinsic"}) or embedded (\code{"extrinsic"}) geometry.}

\item{...}{extra parameters including\describe{
\item{p}{an exponent (default: 2).}
}}
}
\value{
the distance value.
}
\description{
Given two curves \eqn{\gamma_1, \gamma_2 : I \rightarrow \mathcal{M}}, we are 
interested in measuring the discrepancy of two curves. Usually, data are given 
as discrete observations so we are offering several methods to perform the task. See 
the section below for detailed description.
}
\section{Default Method }{
 Trapezoidal Approximation
Assume \eqn{\gamma_1 (t_i) = X_i} and \eqn{\gamma_2 (t_i) = Y_i} for 
\eqn{i=1,2,\ldots,N}. In the Euclidean space, \eqn{L_p} distance between two 
scalar-valued functions is defined as 
\deqn{L_p^p (\gamma_1 (x), \gamma_2 (x) = \int_{\mathcal{X}} |\gamma_1 (x) - \gamma_2 (x)|^p dx }. 
We extend this approach to manifold-valued curves
\deqn{L_p^p (\gamma_1 (t), \gamma_2 (t)) = \int_{t\in I} d^p (\gamma_1 (t), \gamma_2 (t)) dt}
where \eqn{d(\cdot,\cdot)} is an intrinsic/extrinsic distance on manifolds. With the given 
representations, the above integral is approximated using trapezoidal rule.
}

\examples{
#-------------------------------------------------------------------
#                          Curves on Sphere
#
#  curve1 : y = 0.5*cos(x) on the tangent space at (0,0,1)
#  curve2 : y = 0.5*cos(x) on the tangent space at (0,0,1)
#  curve3 : y = 0.5*sin(x) on the tangent space at (0,0,1)
#
# * distance between curve1 & curve2 should be close to 0.
# * distance between curve1 & curve3 should be large.
#-------------------------------------------------------------------
## GENERATION
vecx  = seq(from=-0.9, to=0.9, length.out=50)
vecy1 = 0.5*cos(vecx) + rnorm(50, sd=0.05)
vecy2 = 0.5*cos(vecx) + rnorm(50, sd=0.05)
vecy3 = 0.5*sin(vecx) + rnorm(50, sd=0.05)

## WRAP AS RIEMOBJ
mat1 = cbind(vecx, vecy1, 1); mat1 = mat1/sqrt(rowSums(mat1^2))
mat2 = cbind(vecx, vecy2, 1); mat2 = mat2/sqrt(rowSums(mat2^2))
mat3 = cbind(vecx, vecy3, 1); mat3 = mat3/sqrt(rowSums(mat3^2))

rcurve1 = wrap.sphere(mat1)
rcurve2 = wrap.sphere(mat2)
rcurve3 = wrap.sphere(mat3)

## COMPUTE DISTANCES
riem.distlp(rcurve1, rcurve2, vect=vecx)
riem.distlp(rcurve1, rcurve3, vect=vecx)

}
\concept{curve}
