#%%%%%%%%%%%%%%%%%%%%%
# GLOBAL LIBRARY #####
#%%%%%%%%%%%%%%%%%%%%%

require(DT)
require(plotly)
require(shinyjs)
require(shinyBS)

#%%%%%%%%%%%%%%%%%%%%%
# SOURCING ###########
#%%%%%%%%%%%%%%%%%%%%%

source("ui/About.R", local = T)
source("ui/Data.R", local = T)
source("ui/IRT.R", local = T)

#%%%%%%%%%%%%%%%%%%%%%
# UI #################
#%%%%%%%%%%%%%%%%%%%%%

ui = tagList(
  tags$head(tags$link(rel = "stylesheet",
                      type = "text/css",
                      href = "style.css"),
            tags$script(type = "text/javascript",
                        src = "busy.js"),
            tags$script(type = "text/javascript",
                        src = "report_generating_message.js"),
            tags$script(type = "text/javascript",
                        src = "report_downloading_message.js"),
            tags$link(rel = "stylesheet",
                      type = "text/css",
                      href = "margins_and_paddings.css"),
            tags$link(rel = "shortcut icon", href = "hexbin.png"),
            tags$style(type = "text/css",
                       ".panel-footer {position: fixed; right: 0; bottom: 0; left: 0;}")),
  div(class = "busy",
      p("Loading"),
      img(src = "busy_indicator.gif", height = 100, width = 100)
  ),

  shinyjs::useShinyjs(),

  tags$head(includeScript("google-analytics.js")),

  navbarPage(title = div(HTML('<font size = "5"> ShinyItemAnalysis </font>
                               <font size = "2"> Test and item analysis </font> </p>')),
             windowTitle = 'ShinyItemAnalysis',
             position = 'fixed-top',
             selected = 'About',
             collapsible = TRUE,
             footer = list(
               div(class = "clear"),
               div(class = "panel-footer",
                   HTML('<img src = "hexbin.png", style = "float:left; width: 65px;">
                         <p style = "margin:8px 0 0 0;">
                             <font size = "4"> ShinyItemAnalysis </font>
                             <font size = "2"> Test and item analysis | Version 1.2.7 </font>
                             <span style = "float:right">
                                <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysis/" id = "tooltipweb" target="_blank"> <img src = "footer_web_icon.png", style = "width: 25px;"> </a>
                                <a href = "https://github.com/patriciamar/ShinyItemAnalysis/" id = "tooltipgithub" target="_blank"> <img src = "footer_github_icon.png", style = "width: 25px;"> </a>
                                <a href = "https://CRAN.R-project.org/package=ShinyItemAnalysis/" id = "tooltipcran" target="_blank"> <img src = "footer_cran_icon.png", style = "width: 25px;"> </a>
                             </span> </p>
                         <script>
                           $("#tooltipweb").attr("title", "Web");
                           $("#tooltipgithub").attr("title", "GitHub");
                           $("#tooltipcran").attr("title", "CRAN");
                         </script>'),
                   p(`style` = "margin:0 0 0 0;",
                     HTML('<font size = "2"> &copy; 2018  Patricia Martinkova, Adela Drabinova, Ondrej Leder and Jakub Houdek </font>'),
                     HTML('<span style = "float:right"> '),
                     textOutput('counter', inline = T),
                     HTML('</span>')))),
             theme = "bootstrap.css",
             #%%%%%%%%%%%%%%%%%%%%%
             # MAIN PANEL #########
             #%%%%%%%%%%%%%%%%%%%%%

             #%%%%%%%%%%%%%%%%%%%%%
             # ABOUT ##############
             #%%%%%%%%%%%%%%%%%%%%%
             About,
             #%%%%%%%%%%%%%%%%%%%%%
             # DATA ###############
             #%%%%%%%%%%%%%%%%%%%%%
             Data,
             #%%%%%%%%%%%%%%%%%%%%%
             # SUMMARY ############
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Summary",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Analysis of total scores"),
                                 h4("Summary table"),
                                 p("Table below summarizes basic characteristics of total scores including
                                   minimum and maximum, mean, median, standard deviation, skewness and kurtosis.
                                   The kurtosis here is estimated by sample kurtosis \\(\\frac{m_4}{m_2^2}\\),
                                   where \\(m_4\\) is the fourth central moment and \\(m_2\\) is the second
                                   central moment. The skewness is estimated by sample skewness
                                   \\(\\frac{m_3}{m_2^{3/2}}\\), where \\(m_3\\) is the third central moment.
                                   The kurtosis for normally distributed scores is near the value of 3 and the skewness
                                   is near the value of 0. "),
                                 tableOutput('totalscores_table'),
                                 h4("Histogram of total score"),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "slider_totalscores_histogram",
                                                           label = "Cut-score",
                                                           min = 0,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1))),
                                 p('For selected cut-score, blue part of histogram shows respondents with total score
                                   above the cut-score, grey column shows respondents with total score equal
                                   to the cut-score and red part of histogram shows respondents below the cut-score.'),
                                 plotOutput('totalscores_histogram'),
                                 downloadButton(outputId = "DB_totalscores_histogram", label = "Download figure"),
                                 br(),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)<br>library(ggplot2)<br>library(moments)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]<br><br>#&nbsp;total&nbsp;score&nbsp;calculation<br>score&nbsp;<-&nbsp;apply(data,&nbsp;1,&nbsp;sum)<br><br>#&nbsp;summary&nbsp;of&nbsp;total&nbsp;score&nbsp;<br>c(min(score),&nbsp;max(score),&nbsp;mean(score),&nbsp;median(score),&nbsp;sd(score),&nbsp;skewness(score),&nbsp;kurtosis(score))<br><br>#&nbsp;colors&nbsp;by&nbsp;cut-score<br>cut&nbsp;<-&nbsp;median(score)&nbsp;#&nbsp;cut-score&nbsp;<br>color&nbsp;<-&nbsp;c(rep(\"red\",&nbsp;cut&nbsp;-&nbsp;min(score)),&nbsp;\"gray\",&nbsp;rep(\"blue\",&nbsp;max(score)&nbsp;-&nbsp;cut))<br>df&nbsp;<-&nbsp;data.frame(score)<br><br>#&nbsp;histogram<br>ggplot(df,&nbsp;aes(score))&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_histogram(binwidth&nbsp;=&nbsp;1,&nbsp;fill&nbsp;=&nbsp;color,&nbsp;col&nbsp;=&nbsp;\"black\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;xlab(\"Total&nbsp;score\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;ylab(\"Number&nbsp;of&nbsp;respondents\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme_bw()&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme(legend.title&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;axis.line&nbsp;&nbsp;=&nbsp;element_line(colour&nbsp;=&nbsp;\"black\"),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.major&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.minor&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;text&nbsp;=&nbsp;element_text(size&nbsp;=&nbsp;14))"))),
                                 br()
                                 ),
                        # * STANDARD SCORES ####
                        tabPanel("Standard scores",
                                 h3('Standard scores'),
                                 p(strong('Total score'), 'also known as raw score is a total number of correct
                                   answers. It can be used to compare individual score to a norm group, e.g. if the mean
                                   is 12, then individual score can be compared to see if it is below or above this average. ', br(),
                                   strong('Percentile'), 'indicates the value below which a percentage of observations
                                   falls, e.g. a individual score at the 80th percentile means that the individual score
                                   is the same or higher than the scores of 80% of all respondents. ', br(),
                                   strong('Success rate'), 'is the percentage of success, e.g. if the maximum points of test
                                   is equal to 20 and individual score is 12 then success rate is 12/20 = 0.6, i.e. 60%.', br(),
                                   strong('Z-score'), 'or also standardized score is a linear transformation of total
                                   score with a mean of 0 and with variance of 1. If X is total score, M its mean and SD its
                                   standard deviation then Z-score = (X - M) / SD. ', br(),
                                   strong('T-score'), 'is transformed Z-score with a mean of 50 and standard deviation
                                   of 10. If Z is Z-score then T-score = (Z * 10) + 50. '),
                                 h4("Table by score"),
                                 tableOutput('scores_tables'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br><br>#&nbsp;scores&nbsp;calculations<br>score&nbsp;<-&nbsp;apply(data,&nbsp;1,&nbsp;sum)&nbsp;#&nbsp;Total&nbsp;score&nbsp;<br>tosc&nbsp;<-&nbsp;sort(unique(score))&nbsp;#&nbsp;Levels&nbsp;of&nbsp;total&nbsp;score&nbsp;<br>perc&nbsp;<-&nbsp;cumsum(prop.table(table(score)))&nbsp;#&nbsp;Percentiles&nbsp;<br>sura&nbsp;<-&nbsp;100&nbsp;*&nbsp;(tosc&nbsp;/&nbsp;max(score))&nbsp;#&nbsp;Success&nbsp;rate&nbsp;<br>zsco&nbsp;<-&nbsp;sort(unique(scale(score)))&nbsp;#&nbsp;Z-score&nbsp;<br>tsco&nbsp;<-&nbsp;50&nbsp;+&nbsp;10&nbsp;*&nbsp;zsco&nbsp;#&nbsp;T-score"))),
                                 br()
                        )
                        ),
             #%%%%%%%%%%%%%%%%%%%%%
             # VALIDITY ###########
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Validity",
                        # * CORRELATION STRUCTURE ####
                         tabPanel("Correlation structure",
                                  h3("Correlation structure"),
                                  h4("Polychoric correlation heat map"),
                                  p('Polychoric correlation heat map is a correlation plot which displays a polychoric
                                     correlations of items. The size and shade of circles indicate how much the
                                     items are correlated (larger and darker circle means larger correlation).
                                     The color of circles indicates in which way the items are correlated - blue
                                     color shows possitive correlation and red color shows negative correlation.'),
                                  p("Polychoric correlation heat map can be reordered using hierarchical",
                                    HTML("<b>clustering method</b>"), "below.
                                    Ward's method aims at finding compact clusters based on minimizing the within-cluster
                                    sum of squares.
                                    Ward's n. 2 method used squared disimilarities.
                                    Single method connects clusters with the nearest neighbours, i.e. the distance between
                                    two clusters is calculated as the minimum of distances of observations in one cluster and
                                    observations in the other clusters.
                                    Complete linkage with farthest neighbours, i.e. maximum of distances.
                                    Average linkage method used the distance based on weighted average of the individual distances.
                                    With McQuitty method used unweighted average.
                                    Median linkage calculates the distance as the median of distances between an observation
                                    in one cluster and observation in the other cluster.
                                    Centroid method used distance between centroids of clusters. "),
                                  p("With", HTML("<b>number  of clusters</b>"), "larger than 1, the rectangles representing
                                    clusters are drawn. "),
                                  fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                numericInput(inputId = 'corr_plot_clust',
                                                             label = 'Number of clusters',
                                                             value = 1,
                                                             min = 1,
                                                             max = 1)),
                                            div(style = "display: inline-block; vertical-align: top; width: 5%;"),
                                            div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                selectInput(inputId = 'corr_plot_clustmethod',
                                                            label = 'Clustering method',
                                                            choices = list("None" = "none",
                                                                           "Ward's"  = "ward.D",
                                                                           "Ward's n. 2" = "ward.D2",
                                                                           "Single" = "single",
                                                                           "Complete" = "complete",
                                                                           "Average" = "average",
                                                                           "McQuitty" = "mcquitty",
                                                                           "Median" = "median",
                                                                           "Centroid" = "centroid"),
                                                            selected = "none"))),
                                  plotOutput('corr_plot'),
                                  downloadButton(outputId = "DB_corr_plot", label = "Download figure"),
                                  br(),
                                  h4("Scree plot"),
                                  p('A scree plot displays the eigenvalues associated with an component or a factor in descending order
                                     versus the number of the component or factor. '),
                                  plotOutput('scree_plot'),
                                  downloadButton(outputId = "DB_scree_plot", label = "Download figure"),
                                  h4("Selected R code"),
                                  div(code(HTML("library(corrplot)&nbsp;<br>library(difNLR)&nbsp;<br>library(psych)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br><br>#&nbsp;correlation&nbsp;heat&nbsp;map&nbsp;<br>corP&nbsp;<-&nbsp;polychoric(data)&nbsp;#&nbsp;polychoric&nbsp;correlation&nbsp;calculation<br>corP$rho&nbsp;#&nbsp;correlation&nbsp;matrix&nbsp;<br>corrplot(corP$rho)&nbsp;#&nbsp;correlation&nbsp;plot&nbsp;<br>corrplot(corP$rho,&nbsp;order&nbsp;=&nbsp;\"hclust\",&nbsp;hclust.method&nbsp;=&nbsp;\"ward.D\",&nbsp;addrect&nbsp;=&nbsp;3)&nbsp;#&nbsp;correlation&nbsp;plot&nbsp;with&nbsp;3&nbsp;clusters&nbsp;using&nbsp;Ward&nbsp;method<br><br>#&nbsp;scree&nbsp;plot&nbsp;<br>ev&nbsp;<-&nbsp;eigen(corP$rho)$values&nbsp;#&nbsp;eigen&nbsp;values<br>df&nbsp;<-&nbsp;data.frame(comp&nbsp;=&nbsp;1:length(ev),&nbsp;ev)<br><br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;comp,&nbsp;y&nbsp;=&nbsp;ev))&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_point()&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_line()&nbsp;+&nbsp;<br>&nbsp;&nbsp;ylab(\"Eigen&nbsp;value\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;xlab(\"Component&nbsp;number\")&nbsp;+<br>&nbsp;&nbsp;theme_bw()&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme(legend.title&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;axis.line&nbsp;&nbsp;=&nbsp;element_line(colour&nbsp;=&nbsp;\"black\"),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.major&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.minor&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;text&nbsp;=&nbsp;element_text(size&nbsp;=&nbsp;14))"))),
                                  br()
                                    ),
                        # * PREDICTIVE VALIDITY ####
                        tabPanel('Criterion validity',
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Criterion validity'),
                                            p('This section requires criterion variable (e.g. future study success or future GPA in case
                                              of admission tests) which should correlate with the measurement. Criterion variable
                                              can be uploaded in ', strong('Data'), 'section.'),
                                            h4('Descriptive plots of criterion variable on total score'),
                                            p('Total scores are plotted according to criterion variable. Boxplot or scatterplot is displayed
                                              depending on the type of criterion variable - whether it is discrete or continuous. Scatterplot is
                                              provided with red linear regression line. '),
                                            plotOutput('validity_plot'),
                                            downloadButton(outputId = "DB_validity_plot", label = "Download figure"),
                                            h4('Correlation of criterion variable and total score'),
                                            p('Test for association between total score and criterion variable is based on Spearman`s \\(\\rho\\).
                                              This rank-based measure has been recommended if bivariate normal distribution is not guaranteed.
                                              The null hypothesis is that correlation is 0. '),
                                            tableOutput('validity_table'),
                                            htmlOutput('validity_table_interpretation'),
                                            h4("Selected R code"),
                                            div(code(HTML("library(ShinyItemAnalysis)&nbsp;<br>library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data01&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>#&nbsp;total&nbsp;score&nbsp;calculation<br>score&nbsp;<-&nbsp;apply(data01,&nbsp;1,&nbsp;sum)&nbsp;<br>#&nbsp;criterion&nbsp;variable<br>criterion&nbsp;<-&nbsp;GMAT[,&nbsp;\"criterion\"]&nbsp;<br>#&nbsp;number&nbsp;of&nbsp;respondents&nbsp;in&nbsp;each&nbsp;criterion&nbsp;level<br>size&nbsp;<-&nbsp;as.factor(criterion)<br>levels(size)&nbsp;<-&nbsp;table(as.factor(criterion))<br>size&nbsp;<-&nbsp;as.numeric(paste(size))<br>df&nbsp;<-&nbsp;data.frame(score,&nbsp;criterion,&nbsp;size)<br><br>#&nbsp;descriptive&nbsp;plots&nbsp;<br>###&nbsp;boxplot,&nbsp;for&nbsp;discrete&nbsp;criterion<br>ggplot(df,&nbsp;aes(y&nbsp;=&nbsp;score,&nbsp;x&nbsp;=&nbsp;as.factor(criterion),&nbsp;fill&nbsp;=&nbsp;as.factor(criterion)))&nbsp;+<br>&nbsp;&nbsp;geom_boxplot()&nbsp;+<br>&nbsp;&nbsp;geom_jitter(shape&nbsp;=&nbsp;16,&nbsp;position&nbsp;=&nbsp;position_jitter(0.2))&nbsp;+<br>&nbsp;&nbsp;scale_fill_brewer(palette&nbsp;=&nbsp;\"Blues\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Criterion&nbsp;group\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;coord_flip()&nbsp;+<br>&nbsp;&nbsp;theme_bw()&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme(legend.title&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;axis.line&nbsp;&nbsp;=&nbsp;element_line(colour&nbsp;=&nbsp;\"black\"),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.major&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.minor&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;text&nbsp;=&nbsp;element_text(size&nbsp;=&nbsp;14))<br><br>###&nbsp;scatterplot,&nbsp;for&nbsp;continuous&nbsp;criterion<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;score,&nbsp;y&nbsp;=&nbsp;criterion))&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_point()&nbsp;+&nbsp;<br>&nbsp;&nbsp;ylab(\"Criterion&nbsp;variable\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;xlab(\"Total&nbsp;score\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;geom_smooth(method&nbsp;=&nbsp;lm,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;se&nbsp;=&nbsp;FALSE,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"red\")&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme_bw()&nbsp;+&nbsp;<br>&nbsp;&nbsp;theme(legend.title&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;axis.line&nbsp;&nbsp;=&nbsp;element_line(colour&nbsp;=&nbsp;\"black\"),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.major&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;panel.grid.minor&nbsp;=&nbsp;element_blank(),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;text&nbsp;=&nbsp;element_text(size&nbsp;=&nbsp;14))<br><br>#&nbsp;correlation&nbsp;<br>cor.test(criterion,&nbsp;score,&nbsp;method&nbsp;=&nbsp;\"spearman\",&nbsp;exact&nbsp;=&nbsp;F)"))),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Criterion validity'),
                                            p('This section requires criterion variable (e.g. future study success or future GPA in case
                                              of admission tests) which should correlate with the measurement. Criterion variable
                                              can be uploaded in ', strong('Data'), 'section. Here you can explore how the criterion correlates with individual items. '),
                                            p('In distractor analysis based on criterion variable, we are interested in how test takers
                                              select the correct answer and how the distractors (wrong answers) with respect to group based
                                              on criterion variable.'),
                                            h4('Distractor plot'),
                                            htmlOutput("validity_distractor_text"),
                                            p('With option ', strong('Combinations'), 'all item selection patterns are plotted (e.g. AB, ACD, BC). With
                                              option', strong('Distractors'), 'answers are splitted into distractors (e.g. A, B, C, D).'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = 'validity_group',
                                                                      label = 'Number of groups:',
                                                                      min   = 1,
                                                                      max   = 5,
                                                                      value = 3)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 15%; ",
                                                          radioButtons(inputId = 'type_validity_combinations_distractor',
                                                                       label = 'Type',
                                                                       choices = list("Combinations", "Distractors"))),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "validitydistractorSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            plotOutput('validity_distractor_plot'),
                                            downloadButton(outputId = "DB_validity_distractor_plot", label = "Download figure"),
                                            h4('Correlation of criterion variable and scored item'),
                                            p('Test for association between total score and criterion variable is based on Spearman`s \\(\\rho\\).
                                              This rank-based measure has been recommended if bivariate normal distribution is not guaranteed.
                                              The null hypothesis is that correlation is 0. '),
                                            tableOutput('validity_table_item'),
                                            htmlOutput('validity_table_item_interpretation'),
                                            h4("Selected R code"),
                                            div(code(HTML("library(ShinyItemAnalysis)&nbsp;<br>library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(\"GMAT\",&nbsp;\"GMATtest\",&nbsp;\"GMATkey\")&nbsp;<br>data&nbsp;<-&nbsp;GMATtest[,&nbsp;1:20]&nbsp;<br>data01&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>key&nbsp;<-&nbsp;GMATkey&nbsp;<br>criterion&nbsp;<-&nbsp;GMAT[,&nbsp;\"criterion\"]&nbsp;<br><br>#&nbsp;distractor&nbsp;plot&nbsp;for&nbsp;item&nbsp;1&nbsp;and&nbsp;3&nbsp;groups&nbsp;<br>plotDistractorAnalysis(data,&nbsp;key,&nbsp;num.groups&nbsp;=&nbsp;3,&nbsp;item&nbsp;=&nbsp;1,&nbsp;matching&nbsp;=&nbsp;criterion)&nbsp;<br><br>#&nbsp;correlation&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>cor.test(criterion,&nbsp;data01[,&nbsp;1],&nbsp;method&nbsp;=&nbsp;\"spearman\",&nbsp;exact&nbsp;=&nbsp;F)"))),
                                            br()
                                            )
                                   ))),

             #%%%%%%%%%%%%%%%%%%%%%
             # ITEM ANALYSIS ######
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu('Item analysis',
                        # * TRADITIONAL ITEM ANALYSIS ####
                        tabPanel("Traditional item analysis",
                                 h3("Traditional item analysis"),
                                 p('Traditional item analysis uses proportions of correct answers or correlations to estimate item properties.'),
                                 h4("Item difficulty/discrimination plot"),
                                 p("Displayed is difficulty (red) and discrimination (blue)
                                   for all items. Items are ordered by difficulty. ", br(),
                                   strong("Difficulty"),' of items is estimated as percent of respondents who
                                   answered correctly to that item.', br(),
                                   strong("Discrimination"),' is by default described by difference of percent correct
                                   in upper and lower third of respondents (Upper-Lower Index, ULI). By rule of
                                   thumb it should not be lower than 0.2 (borderline in the plot), except for
                                   very easy or very difficult items. Discrimination can be customized (see also Martinkova, Stepanek, et al.
                                   (2017)) by changing number of groups and by changing which groups should be compared: '),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = 'DDplotNumGroupsSlider',
                                                           label = 'Number of groups:',
                                                           min   = 1,
                                                           max   = 5,
                                                           value = 3)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "DDplotRangeSlider",
                                                           label = "Which two groups to compare:",
                                                           min = 1,
                                                           max = 3,
                                                           step = 1,
                                                           value = c(1, 3)))),
                                 htmlOutput("DDplot_text"),
                                 br(),
                                 plotOutput('DDplot'),
                                 downloadButton("DB_DDplot", label = "Download figure"),
                                 h4("Cronbach's alpha"),
                                 p("Chronbach's alpha is an estimate of the reliability of a psychometric test. It is a function
                                   of the number of items in a test, the average covariance between item-pairs, and the variance
                                   of the total score (Cronbach, 1951)."),
                                 tableOutput('cronbachalpha_table'),
                                 h4("Traditional item analysis table"),
                                 htmlOutput("itemanalysis_table_text"),
                                 tableOutput('itemanalysis_table'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(psych)&nbsp;<br>library(psychometric)&nbsp;<br>library(ShinyItemAnalysis)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br><br>#&nbsp;difficulty&nbsp;and&nbsp;discrimination&nbsp;plot&nbsp;<br>DDplot(data,&nbsp;k&nbsp;=&nbsp;3,&nbsp;l&nbsp;=&nbsp;1,&nbsp;u&nbsp;=&nbsp;3)&nbsp;<br><br>#&nbsp;Cronbach&nbsp;alpha&nbsp;<br>psych::alpha(data)&nbsp;<br><br>#&nbsp;traditional&nbsp;item&nbsp;analysis&nbsp;table&nbsp;<br>tab&nbsp;<-&nbsp;round(data.frame(item.exam(data,&nbsp;discr&nbsp;=&nbsp;TRUE)[,&nbsp;c(4,&nbsp;1,&nbsp;5,&nbsp;2,&nbsp;3)],&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;psych::alpha(data)$alpha.drop[,&nbsp;1],&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;gDiscrim(data,&nbsp;k&nbsp;=&nbsp;3,&nbsp;l&nbsp;=&nbsp;1,&nbsp;u&nbsp;=&nbsp;3)),&nbsp;2)&nbsp;<br>colnames(tab)&nbsp;<-&nbsp;c(\"Difficulty\",&nbsp;\"SD\",&nbsp;\"Dsicrimination&nbsp;ULI\",&nbsp;\"Discrimination&nbsp;RIT\",&nbsp;\"Discrimination&nbsp;RIR\",&nbsp;\"Alpha&nbsp;Drop\",&nbsp;\"Customized&nbsp;Discrimination\")&nbsp;<br>tab"))),
                                 br()
                                 ),
                        # * DISTRACTORS ####
                        tabPanel("Distractors",
                                 h3("Distractor analysis"),
                                 p('In distractor analysis, we are interested in how test takers select
                                   the correct answer and how the distractors (wrong answers) were able
                                   to function effectively by drawing the test takers away from the correct answer.'),
                                 h4("Distractors plot"),
                                 htmlOutput("distractor_text"),
                                 p('With option ', strong('Combinations'), 'all item selection patterns are plotted (e.g. AB, ACD, BC). With
                                   option', strong('Distractors'), 'answers are splitted into distractors (e.g. A, B, C, D).'),
                                 fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = 'gr',
                                                           label = 'Number of groups:',
                                                           min   = 1,
                                                           max   = 5,
                                                           value = 3)),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 15%; ",
                                               radioButtons(inputId = 'type_combinations_distractor',
                                                            label = 'Type',
                                                            choices = list("Combinations", "Distractors"))),
                                           div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                           div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                               sliderInput(inputId = "distractorSlider",
                                                           label = "Item",
                                                           min = 1,
                                                           max = 10,
                                                           value = 1,
                                                           step = 1,
                                                           animate = TRUE))),
                                 plotOutput('distractor_plot'),
                                 downloadButton("DB_distractor_plot", label = "Download figure"),
                                 br(),
                                 h4("Table with counts"),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_counts'))),
                                 h4("Table with proportions"),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_proportions'))),
                                 br(),
                                 h4('Barplot of item response patterns'),
                                 plotOutput("distractor_barplot_item_response_patterns"),
                                 downloadButton( "DB_distractor_barplot_item_response_patterns", label = "Download figure"),
                                 h4('Histogram of total scores'),
                                 plotOutput('distractor_histogram'),
                                 downloadButton("DB_distractor_histogram", label = "Download figure"),
                                 br(),
                                 h4('Table of total scores by groups'),
                                 fluidRow(column(12, align = "center", tableOutput('distractor_table_total_score_by_group'))),
                                 br(),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ShinyItemAnalysis)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMATtest)&nbsp;<br>data&nbsp;<-&nbsp;GMATtest[,&nbsp;1:20]&nbsp;<br>data(GMATkey)&nbsp;<br>key&nbsp;<-&nbsp;GMATkey&nbsp;<br><br>#&nbsp;combinations&nbsp;-&nbsp;plot&nbsp;for&nbsp;item&nbsp;1&nbsp;and&nbsp;3&nbsp;groups&nbsp;<br>plotDistractorAnalysis(data,&nbsp;key,&nbsp;num.group&nbsp;=&nbsp;3,&nbsp;item&nbsp;=&nbsp;1,&nbsp;multiple.answers&nbsp;=&nbsp;T)&nbsp;<br><br>#&nbsp;distractors&nbsp;-&nbsp;plot&nbsp;for&nbsp;item&nbsp;1&nbsp;and&nbsp;3&nbsp;groups&nbsp;<br>plotDistractorAnalysis(data,&nbsp;key,&nbsp;num.group&nbsp;=&nbsp;3,&nbsp;item&nbsp;=&nbsp;1,&nbsp;multiple.answers&nbsp;=&nbsp;F)&nbsp;<br><br>#&nbsp;table&nbsp;with&nbsp;counts&nbsp;and&nbsp;margins&nbsp;-&nbsp;item&nbsp;1&nbsp;and&nbsp;3&nbsp;groups&nbsp;<br>DA&nbsp;<-&nbsp;DistractorAnalysis(data,&nbsp;key,&nbsp;num.groups&nbsp;=&nbsp;3)[[1]]&nbsp;<br>dcast(as.data.frame(DA),&nbsp;response&nbsp;~&nbsp;score.level,&nbsp;sum,&nbsp;margins&nbsp;=&nbsp;T,&nbsp;value.var&nbsp;=&nbsp;\"Freq\")&nbsp;<br><br>#&nbsp;table&nbsp;with&nbsp;proportions&nbsp;-&nbsp;item&nbsp;1&nbsp;and&nbsp;3&nbsp;groups&nbsp;<br>DistractorAnalysis(data,&nbsp;key,&nbsp;num.groups&nbsp;=&nbsp;3,&nbsp;p.table&nbsp;=&nbsp;T)[[1]]"))),
                                 br()
                                 )
                        ),

             #%%%%%%%%%%%%%%%%%%%%%
             # REGRESSION #########
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("Regression",
                        "Dichotomous models",
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 h3("Logistic regression on total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'),'can model dependency of probability of correct answer on total score by
                                   S-shaped logistic curve. Parameter', strong( "b0"),' describes horizontal position of the fitted curve,
                                   parameter ', strong( 'b1'),' describes its slope.'),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to total score.
                                   Their size is determined by count of respondents who achieved given level of
                                   total score.'),
                                 sliderInput("logregSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('logreg_plot'),
                                 downloadButton("DB_logreg_plot", label = "Download figure"),
                                 h4("Equation"),
                                 withMathJax(),
                                 ('$$\\mathrm{P}(Y = 1|X, b_0, b_1) = \\mathrm{E}(Y|X, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 X\\right)}}{1+e^{\\left( b_{0} + b_1 X\\right) }} $$'),

                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('logreg_table'))),
                                 htmlOutput("logreg_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ggplot2)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>score&nbsp;<-&nbsp;apply(data,&nbsp;1,&nbsp;sum)&nbsp;#&nbsp;total&nbsp;score<br><br>#&nbsp;logistic&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit&nbsp;<-&nbsp;glm(data[,&nbsp;1]&nbsp;~&nbsp;score,&nbsp;family&nbsp;=&nbsp;binomial)&nbsp;<br><br>#&nbsp;coefficients&nbsp;<br>coef(fit)&nbsp;<br><br>#&nbsp;function&nbsp;for&nbsp;plot&nbsp;<br>fun&nbsp;<-&nbsp;function(x,&nbsp;b0,&nbsp;b1){exp(b0&nbsp;+&nbsp;b1&nbsp;*&nbsp;x)&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(b0&nbsp;+&nbsp;b1&nbsp;*&nbsp;x))}&nbsp;<br><br>#&nbsp;empirical&nbsp;probabilities&nbsp;calculation<br>df&nbsp;<-&nbsp;data.frame(x&nbsp;=&nbsp;sort(unique(score)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y&nbsp;=&nbsp;tapply(data[,&nbsp;1],&nbsp;score,&nbsp;mean),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;as.numeric(table(score)))<br><br>#&nbsp;plot&nbsp;of&nbsp;estimated&nbsp;curve<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;x,&nbsp;y&nbsp;=&nbsp;y))&nbsp;+<br>&nbsp;&nbsp;geom_point(aes(size&nbsp;=&nbsp;size),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;fill&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shape&nbsp;=&nbsp;21,&nbsp;alpha&nbsp;=&nbsp;0.5)&nbsp;+<br>&nbsp;&nbsp;stat_function(fun&nbsp;=&nbsp;fun,&nbsp;geom&nbsp;=&nbsp;\"line\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;args&nbsp;=&nbsp;list(b0&nbsp;=&nbsp;coef(fit)[1],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;b1&nbsp;=&nbsp;coef(fit)[2]),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;1,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Probability&nbsp;of&nbsp;correct&nbsp;answer\")&nbsp;+<br>&nbsp;&nbsp;ylim(0,&nbsp;1)&nbsp;+<br>&nbsp;&nbsp;ggtitle(\"Item&nbsp;1\")"))),
                                 br()
                                 ),
                        # * LOGISTIC Z ####
                        tabPanel("Logistic Z",
                                 h3("Logistic regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by S-shaped logistic curve. Parameter ', strong( 'b0'), ' describes
                                   horizontal position of the fitted curve (difficulty), parameter ', strong('b1'),' describes its slope at
                                   inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput("zlogregSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('z_logreg_plot'),
                                 downloadButton("DB_z_logreg_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1) = \\mathrm{E}(Y|Z, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 Z\\right) }}{1+e^{\\left( b_{0} + b_1 Z\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('z_logreg_table'))),
                                 htmlOutput("z_logreg_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ggplot2)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(data,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br><br>#&nbsp;logistic&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit&nbsp;<-&nbsp;glm(data[,&nbsp;1]&nbsp;~&nbsp;zscore,&nbsp;family&nbsp;=&nbsp;binomial)&nbsp;<br><br>#&nbsp;coefficients&nbsp;<br>coef(fit)&nbsp;<br><br>#&nbsp;function&nbsp;for&nbsp;plot&nbsp;<br>fun&nbsp;<-&nbsp;function(x,&nbsp;b0,&nbsp;b1){exp(b0&nbsp;+&nbsp;b1&nbsp;*&nbsp;x)&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(b0&nbsp;+&nbsp;b1&nbsp;*&nbsp;x))}&nbsp;<br><br>#&nbsp;empirical&nbsp;probabilities&nbsp;calculation<br>df&nbsp;<-&nbsp;data.frame(x&nbsp;=&nbsp;sort(unique(zscore)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y&nbsp;=&nbsp;tapply(data[,&nbsp;1],&nbsp;zscore,&nbsp;mean),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;as.numeric(table(zscore)))<br><br>#&nbsp;plot&nbsp;of&nbsp;estimated&nbsp;curve<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;x,&nbsp;y&nbsp;=&nbsp;y))&nbsp;+<br>&nbsp;&nbsp;geom_point(aes(size&nbsp;=&nbsp;size),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;fill&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shape&nbsp;=&nbsp;21,&nbsp;alpha&nbsp;=&nbsp;0.5)&nbsp;+<br>&nbsp;&nbsp;stat_function(fun&nbsp;=&nbsp;fun,&nbsp;geom&nbsp;=&nbsp;\"line\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;args&nbsp;=&nbsp;list(b0&nbsp;=&nbsp;coef(fit)[1],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;b1&nbsp;=&nbsp;coef(fit)[2]),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;1,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Standardized&nbsp;total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Probability&nbsp;of&nbsp;correct&nbsp;answer\")&nbsp;+<br>&nbsp;&nbsp;ylim(0,&nbsp;1)&nbsp;+<br>&nbsp;&nbsp;ggtitle(\"Item&nbsp;1\")"))),
                                 br()
                                 ),
                        # * LOGISTIC IRT Z ####
                        tabPanel("Logistic IRT Z",
                                 h3("Logistic regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. Note change in parametrization - the IRT parametrization
                                   used here corresponds to the parametrization used in IRT models.
                                   Parameter', strong('b') , 'describes horizontal position of the fitted curve (difficulty),
                                   parameter' , strong('a') , ' describes its slope at inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput("zlogreg_irtSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('z_logreg_irt_plot'),
                                 downloadButton("DB_z_logreg_irt_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, a, b) = \\mathrm{E}(Y|Z, a, b) = \\frac{e^{ a\\left(Z - b\\right) }}{1+e^{a\\left(Z - b\\right)}} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('z_logreg_irt_table'))),
                                 htmlOutput("z_logreg_irt_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ggplot2)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(data,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br><br>#&nbsp;logistic&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit&nbsp;<-&nbsp;glm(data[,&nbsp;1]&nbsp;~&nbsp;zscore,&nbsp;family&nbsp;=&nbsp;binomial)&nbsp;<br><br>#&nbsp;coefficients<br>coef&nbsp;<-&nbsp;c(a&nbsp;=&nbsp;coef(fit)[2],&nbsp;b&nbsp;=&nbsp;-&nbsp;coef(fit)[1]&nbsp;/&nbsp;coef(fit)[2])&nbsp;<br>coef&nbsp;&nbsp;<br><br>#&nbsp;function&nbsp;for&nbsp;plot&nbsp;<br>fun&nbsp;<-&nbsp;function(x,&nbsp;a,&nbsp;b){exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b))&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b)))}&nbsp;<br><br>#&nbsp;empirical&nbsp;probabilities&nbsp;calculation<br>df&nbsp;<-&nbsp;data.frame(x&nbsp;=&nbsp;sort(unique(zscore)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y&nbsp;=&nbsp;tapply(data[,&nbsp;1],&nbsp;zscore,&nbsp;mean),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;as.numeric(table(zscore)))<br><br>#&nbsp;plot&nbsp;of&nbsp;estimated&nbsp;curve<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;x,&nbsp;y&nbsp;=&nbsp;y))&nbsp;+<br>&nbsp;&nbsp;geom_point(aes(size&nbsp;=&nbsp;size),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;fill&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shape&nbsp;=&nbsp;21,&nbsp;alpha&nbsp;=&nbsp;0.5)&nbsp;+<br>&nbsp;&nbsp;stat_function(fun&nbsp;=&nbsp;fun,&nbsp;geom&nbsp;=&nbsp;\"line\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;args&nbsp;=&nbsp;list(a&nbsp;=&nbsp;coef[1],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;b&nbsp;=&nbsp;coef[2]),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;1,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Standardized&nbsp;total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Probability&nbsp;of&nbsp;correct&nbsp;answer\")&nbsp;+<br>&nbsp;&nbsp;ylim(0,&nbsp;1)&nbsp;+<br>&nbsp;&nbsp;ggtitle(\"Item&nbsp;1\")"))),
                                 br()
                                 ),

                        # * NONLINEAR 3P IRT Z ####
                        tabPanel("Nonlinear 3P IRT Z",
                                 h3("Nonlinear three parameter regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Nonlinear regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. The IRT parametrization used here corresponds
                                   to the parametrization used in IRT models. Parameter ', strong( 'b'),' describes horizontal position of the fitted curve (difficulty),
                                   parameter ',strong( 'a'), ' describes its slope at inflection point (discrimination). This model allows for nonzero lower left
                                   asymptote ', strong( 'c'), ' (pseudo-guessing parameter). '),
                                 br(),
                                 h4("Plot with estimated nonlinear curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput(inputId = "slider_nlr_3P_item", label = "Item",
                                             min = 1, value = 1, max = 10, step = 1, animate = TRUE),
                                 plotOutput('nlr_3P_plot'),
                                 downloadButton("DB_nlr_3P_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1, c) = \\mathrm{E}(Y|Z, b_0, b_1, c) = c + \\left( 1-c \\right) \\cdot \\frac{e^{a\\left(Z-b\\right) }}{1+e^{a\\left(Z-b\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('nlr_3P_table'))),
                                 htmlOutput("nlr_3P_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ggplot2)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(data,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br><br>#&nbsp;NLR&nbsp;3P&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fun&nbsp;<-&nbsp;function(x,&nbsp;a,&nbsp;b,&nbsp;c){c&nbsp;+&nbsp;(1&nbsp;-&nbsp;c)&nbsp;*&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b))&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b)))}&nbsp;<br><br>fit&nbsp;<-&nbsp;nls(data[,&nbsp;1]&nbsp;~&nbsp;fun(zscore,&nbsp;a,&nbsp;b,&nbsp;c),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;algorithm&nbsp;=&nbsp;\"port\",&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;start&nbsp;=&nbsp;startNLR(data,&nbsp;GMAT[,&nbsp;\"group\"],&nbsp;model&nbsp;=&nbsp;\"3PLcg\",&nbsp;parameterization&nbsp;=&nbsp;\"classic\")[[1]][1:3],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;lower&nbsp;=&nbsp;c(-Inf,&nbsp;-Inf,&nbsp;0,),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;upper&nbsp;=&nbsp;c(Inf,&nbsp;Inf,&nbsp;1))&nbsp;<br>#&nbsp;coefficients&nbsp;<br>coef(fit)&nbsp;<br><br>#&nbsp;empirical&nbsp;probabilities&nbsp;calculation<br>df&nbsp;<-&nbsp;data.frame(x&nbsp;=&nbsp;sort(unique(zscore)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y&nbsp;=&nbsp;tapply(data[,&nbsp;1],&nbsp;zscore,&nbsp;mean),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;as.numeric(table(zscore)))<br><br>#&nbsp;plot&nbsp;of&nbsp;estimated&nbsp;curve<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;x,&nbsp;y&nbsp;=&nbsp;y))&nbsp;+<br>&nbsp;&nbsp;geom_point(aes(size&nbsp;=&nbsp;size),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;fill&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shape&nbsp;=&nbsp;21,&nbsp;alpha&nbsp;=&nbsp;0.5)&nbsp;+<br>&nbsp;&nbsp;stat_function(fun&nbsp;=&nbsp;fun,&nbsp;geom&nbsp;=&nbsp;\"line\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;args&nbsp;=&nbsp;list(a&nbsp;=&nbsp;coef(fit)[1],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;b&nbsp;=&nbsp;coef(fit)[2],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;c&nbsp;=&nbsp;coef(fit)[3]),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;1,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Standardized&nbsp;total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Probability&nbsp;of&nbsp;correct&nbsp;answer\")&nbsp;+<br>&nbsp;&nbsp;ylim(0,&nbsp;1)&nbsp;+<br>&nbsp;&nbsp;ggtitle(\"Item&nbsp;1\")"))),
                                 br()
                                 ),
                        # * NONLINEAR 4P IRT Z ####
                        tabPanel("Nonlinear 4P IRT Z",
                                 h3("Nonlinear four parameter regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Nonlinear four parameter regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. The IRT parametrization used here corresponds
                                   to the parametrization used in IRT models. Parameter ', strong( 'b'),' describes horizontal position of the fitted curve (difficulty),
                                   parameter ', strong( 'a'), ' describes its slope at inflection point (discrimination), pseudo-guessing parameter ', strong('c'), '
                                   is describes lower asymptote and inattention parameter ', strong('d'), 'describes upper asymptote.'),
                                 br(),
                                 h4("Plot with estimated nonlinear curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score.'),
                                 sliderInput(inputId = "slider_nlr_4P_item", label = "Item",
                                             min = 1, value = 1, max = 10, step = 1, animate = TRUE),
                                 plotOutput('nlr_4P_plot'),
                                 downloadButton("DB_nlr_4P_plot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1, c) = \\mathrm{E}(Y|Z, b_0, b_1, c) = c + \\left( d-c \\right) \\cdot \\frac{e^{a\\left(Z-b\\right) }}{1+e^{a\\left(Z-b\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('nlr_4P_table'))),
                                 htmlOutput("nlr_4P_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(ggplot2)<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(data,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br><br>#&nbsp;NLR&nbsp;4P&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fun&nbsp;<-&nbsp;function(x,&nbsp;a,&nbsp;b,&nbsp;c,&nbsp;d){c&nbsp;+&nbsp;(d&nbsp;-&nbsp;c)&nbsp;*&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b))&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b)))}&nbsp;<br><br>fit&nbsp;<-&nbsp;nls(data[,&nbsp;1]&nbsp;~&nbsp;fun(zscore,&nbsp;a,&nbsp;b,&nbsp;c,&nbsp;d),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;algorithm&nbsp;=&nbsp;\"port\",&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;start&nbsp;=&nbsp;startNLR(data,&nbsp;GMAT[,&nbsp;\"group\"],&nbsp;model&nbsp;=&nbsp;\"4PLcgdg\",&nbsp;parameterization&nbsp;=&nbsp;\"classic\")[[1]][1:4],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;lower&nbsp;=&nbsp;c(-Inf,&nbsp;-Inf,&nbsp;0,&nbsp;0),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;upper&nbsp;=&nbsp;c(Inf,&nbsp;Inf,&nbsp;1,&nbsp;1))&nbsp;<br>#&nbsp;coefficients&nbsp;<br>coef(fit)&nbsp;<br><br>#&nbsp;empirical&nbsp;probabilities&nbsp;calculation<br>df&nbsp;<-&nbsp;data.frame(x&nbsp;=&nbsp;sort(unique(zscore)),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y&nbsp;=&nbsp;tapply(data[,&nbsp;1],&nbsp;zscore,&nbsp;mean),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;as.numeric(table(zscore)))<br><br>#&nbsp;plot&nbsp;of&nbsp;estimated&nbsp;curve<br>ggplot(df,&nbsp;aes(x&nbsp;=&nbsp;x,&nbsp;y&nbsp;=&nbsp;y))&nbsp;+<br>&nbsp;&nbsp;geom_point(aes(size&nbsp;=&nbsp;size),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;fill&nbsp;=&nbsp;\"darkblue\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;shape&nbsp;=&nbsp;21,&nbsp;alpha&nbsp;=&nbsp;0.5)&nbsp;+<br>&nbsp;&nbsp;stat_function(fun&nbsp;=&nbsp;fun,&nbsp;geom&nbsp;=&nbsp;\"line\",<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;args&nbsp;=&nbsp;list(a&nbsp;=&nbsp;coef(fit)[1],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;b&nbsp;=&nbsp;coef(fit)[2],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;c&nbsp;=&nbsp;coef(fit)[3],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;d&nbsp;=&nbsp;coef(fit)[4]),<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;size&nbsp;=&nbsp;1,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;color&nbsp;=&nbsp;\"darkblue\")&nbsp;+<br>&nbsp;&nbsp;xlab(\"Standardized&nbsp;total&nbsp;score\")&nbsp;+<br>&nbsp;&nbsp;ylab(\"Probability&nbsp;of&nbsp;correct&nbsp;answer\")&nbsp;+<br>&nbsp;&nbsp;ylim(0,&nbsp;1)&nbsp;+<br>&nbsp;&nbsp;ggtitle(\"Item&nbsp;1\")"))),
                                 br()
                                 ),
                        # * MODELS COMPARISON ####
                        tabPanel("Model comparison",
                                 h3("Logistic regression model selection"),
                                 p('Here you can compare classic 2PL logistic regression model to non-linear model
                                   item by item using some information criteria: '),
                                 tags$ul(
                                   tags$li(strong('AIC'), 'is the Akaike information criterion (Akaike, 1974), '),
                                   tags$li(strong('BIC'), 'is the Bayesian information criterion (Schwarz, 1978)')
                                 ),
                                 p('Another approach to nested models can be likelihood ratio chi-squared test.
                                   Significance level is set to 0.05. As tests are performed item by item, it is
                                   possible to use multiple comparison correction method. '),
                                 selectInput("correction_method_regrmodels", "Correction method",
                                             c("BH" = "BH",
                                               "Holm" = "holm",
                                               "Hochberg" = "hochberg",
                                               "Hommel" = "hommel",
                                               "BY" = "BY",
                                               "FDR" = "fdr",
                                               "none" = "none"),
                                             selected="none"),
                                 h4("Table of comparison statistics"),
                                 p('Rows ', strong('BEST'), 'indicate which model has the lowest value of criterion, or is the largest
                                   significant model by likelihood ratio test.'),
                                 DT::dataTableOutput('regr_comp_table'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT)&nbsp;<br>Data&nbsp;<-&nbsp;GMAT[,&nbsp;1:20]&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(Data,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br><br>#&nbsp;function&nbsp;for&nbsp;fitting&nbsp;models<br>fun&nbsp;<-&nbsp;function(x,&nbsp;a,&nbsp;b,&nbsp;c,&nbsp;d){c&nbsp;+&nbsp;(d&nbsp;-&nbsp;c)&nbsp;*&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b))&nbsp;/&nbsp;(1&nbsp;+&nbsp;exp(a&nbsp;*&nbsp;(x&nbsp;-&nbsp;b)))}&nbsp;<br><br>#&nbsp;starting&nbsp;values&nbsp;for&nbsp;item&nbsp;1<br>start&nbsp;<-&nbsp;startNLR(Data,&nbsp;GMAT[,&nbsp;\"group\"],&nbsp;model&nbsp;=&nbsp;\"4PLcgdg\",&nbsp;parameterization&nbsp;=&nbsp;\"classic\")[[1]][,&nbsp;1:4]<br><br>#&nbsp;2PL&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit2PL&nbsp;<-&nbsp;nls(Data[,&nbsp;1]&nbsp;~&nbsp;fun(zscore,&nbsp;a,&nbsp;b,&nbsp;c&nbsp;=&nbsp;0,&nbsp;d&nbsp;=&nbsp;1),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;algorithm&nbsp;=&nbsp;\"port\",&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;start&nbsp;=&nbsp;start[1:2])&nbsp;<br>#&nbsp;NLR&nbsp;3P&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit3PL&nbsp;<-&nbsp;nls(Data[,&nbsp;1]&nbsp;~&nbsp;fun(zscore,&nbsp;a,&nbsp;b,&nbsp;c,&nbsp;d&nbsp;=&nbsp;1),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;algorithm&nbsp;=&nbsp;\"port\",&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;start&nbsp;=&nbsp;start[1:3],<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;lower&nbsp;=&nbsp;c(-Inf,&nbsp;-Inf,&nbsp;0),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;upper&nbsp;=&nbsp;c(Inf,&nbsp;Inf,&nbsp;1))&nbsp;<br>#&nbsp;NLR&nbsp;4P&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit3PL&nbsp;<-&nbsp;nls(Data[,&nbsp;1]&nbsp;~&nbsp;fun(zscore,&nbsp;a,&nbsp;b,&nbsp;c,&nbsp;d),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;algorithm&nbsp;=&nbsp;\"port\",&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;start&nbsp;=&nbsp;start,<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;lower&nbsp;=&nbsp;c(-Inf,&nbsp;-Inf,&nbsp;0,&nbsp;0),&nbsp;<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;upper&nbsp;=&nbsp;c(Inf,&nbsp;Inf,&nbsp;1,&nbsp;1))&nbsp;<br><br>#&nbsp;comparison&nbsp;<br>###&nbsp;AIC<br>AIC(fit2PL);&nbsp;AIC(fit3PL);&nbsp;AIC(fit4PL)&nbsp;<br>###&nbsp;BIC<br>BIC(fit2PL);&nbsp;BIC(fit3PL);&nbsp;BIC(fit4PL)&nbsp;<br>###&nbsp;LR&nbsp;test,&nbsp;using&nbsp;Benjamini-Hochberg&nbsp;correction<br>######&nbsp;2PL&nbsp;vs&nbsp;NLR&nbsp;3P<br>LRstat&nbsp;<-&nbsp;-2&nbsp;*&nbsp;(sapply(fit2PL,&nbsp;logLik)&nbsp;-&nbsp;sapply(fit3PL,&nbsp;logLik))&nbsp;<br>LRdf&nbsp;<-&nbsp;1&nbsp;<br>LRpval&nbsp;<-&nbsp;1&nbsp;-&nbsp;pchisq(LRstat,&nbsp;LRdf)&nbsp;<br>LRpval&nbsp;<-&nbsp;p.adjust(LRpval,&nbsp;method&nbsp;=&nbsp;\"BH\")&nbsp;<br>######&nbsp;NLR&nbsp;3P&nbsp;vs&nbsp;NLR&nbsp;4P<br>LRstat&nbsp;<-&nbsp;-2&nbsp;*&nbsp;(sapply(fit3PL,&nbsp;logLik)&nbsp;-&nbsp;sapply(fit4PL,&nbsp;logLik))&nbsp;<br>LRdf&nbsp;<-&nbsp;1&nbsp;<br>LRpval&nbsp;<-&nbsp;1&nbsp;-&nbsp;pchisq(LRstat,&nbsp;LRdf)&nbsp;<br>LRpval&nbsp;<-&nbsp;p.adjust(LRpval,&nbsp;method&nbsp;=&nbsp;\"BH\")"))),
                                 br()
                                 ),
                        "----",
                        "Polytomous models",
                        # * MULTINOMIAL ####
                        tabPanel("Multinomial",
                                 h3("Multinomial regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Multinomial regression'),'allows for simultaneous modelling of probability of choosing
                                   given distractors on standardized total score (Z-score).'),
                                 br(),
                                 h4("Plot with estimated curves of multinomial regression"),
                                 p('Points represent proportion of selected option with respect to standardized
                                   total score. Their size is determined by count of respondents who achieved given
                                   level of standardized total score and who selected given option.'),
                                 sliderInput("multiSlider", "Item",
                                             min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 plotOutput('multi_plot'),
                                 downloadButton("DB_multi_plot", label = "Download figure"),
                                 h4("Equation"),
                                 uiOutput('multi_equation'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('multi_table'))),
                                 strong("Interpretation:"),
                                 htmlOutput("multi_interpretation"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code(HTML("library(difNLR)&nbsp;<br>library(nnet)&nbsp;<br><br>#&nbsp;loading&nbsp;data<br>data(GMAT,&nbsp;GMATtest,&nbsp;GMATkey)&nbsp;<br>zscore&nbsp;<-&nbsp;scale(apply(GMAT[,&nbsp;1:20]&nbsp;,&nbsp;1,&nbsp;sum))&nbsp;#&nbsp;standardized&nbsp;total&nbsp;score<br>data&nbsp;<-&nbsp;GMATtest[,&nbsp;1:20]&nbsp;<br>key&nbsp;<-GMATkey<br><br>#&nbsp;multinomial&nbsp;model&nbsp;for&nbsp;item&nbsp;1&nbsp;<br>fit&nbsp;<-&nbsp;multinom(relevel(data[,&nbsp;1],&nbsp;ref&nbsp;=&nbsp;paste(key[1]))&nbsp;~&nbsp;zscore)&nbsp;<br><br>#&nbsp;coefficients&nbsp;<br>coef(fit)"))),
                                 br()
                                 )
                        ),
             #%%%%%%%%%%%%%%%%%%%%%
             # IRT MODELS #########
             #%%%%%%%%%%%%%%%%%%%%%
             IRT,
             #%%%%%%%%%%%%%%%%%%%%%
             # DIF/FAIRNESS #######
             #%%%%%%%%%%%%%%%%%%%%%
             navbarMenu("DIF/Fairness",
                        # * SUMMARY ####
                        "Description",
                        tabPanel('About DIF',
                                 h3('Differential Item Functioning / Item Fairness'),
                                 p('Differential item functioning (DIF) occurs when people from different
                                   social groups (commonly gender or ethnicity) with the same underlying true
                                   ability have a different probability of answering the item correctly.
                                   If item functions differently for two groups, it is potentially unfair.
                                   In general, two type of DIF can be recognized: if the item has different
                                   difficulty for given two groups with the same discrimination, ',
                                   strong('uniform'), 'DIF is present (left figure). If the item has different
                                   discrimination and possibly also different difficulty for given two groups, ',
                                   strong('non-uniform'), 'DIF is present (right figure)'),
                                 br(),
                                 img(src = "fig_DIF_uniform.png",
                                     style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                                 img(src = "fig_DIF_nonuniform.png",
                                     style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                                 br(),
                                 br()
                        ),
                        "----",
                        "Used methods",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Total scores"),
                                 p('DIF is not about total scores! Two groups may have the same distribution of total scores, yet,
                                   some item may function differently for two groups. Also, one of the groups may have signifficantly
                                   lower total score, yet, it may happen that there is no DIF item!',
                                 a('(Martinkova et al., 2017). ',
                                 href = "https://www.lifescied.org/doi/10.1187/cbe.16-10-0307",
                                 target = "_blank")),
                                 h4("Summary of total scores for groups"),
                                 tableOutput('resultsgroup'),
                                 h4("Histograms of total scores for groups"),
                                 sliderInput("inSlider2group", "Cut-score", min = 1, value = 1, max = 10,
                                             step = 1, animate = TRUE),
                                 p('For selected cut-score, blue part of histogram shows respondents with total score
                                   above the cut-score, grey column shows respondents with Total Score equal
                                   to cut-score and red part of histogram shows respondents below the cut-score.'),
                                 splitLayout(cellWidths = c("50%", "50%"), plotOutput('histbyscoregroup0'),plotOutput('histbyscoregroup1')),
                                 splitLayout(cellWidths = c("50%", "50%"), downloadButton("DP_histbyscoregroup0", label = "Download figure"),
                                 downloadButton("DP_histbyscoregroup1", label = "Download figure")),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Summary table'),
                                     br(),
                                     code('sc_zero <- apply(data[group == 0, ], 1, sum); summary(sc_zero) # total scores of reference group'),
                                     br(),
                                     code('sc_one  <- apply(data[group == 1, ], 1, sum); summary(sc_one)  # total scores of focal group'),
                                     br(),
                                     code('# Histograms'),
                                     br(),
                                     code('hist(sc_zero, breaks = 0:20)'),
                                     br(),
                                     code('hist(sc_one, breaks = 0:20)')),
                                 br()
                                 ),
                        # * DELTA PLOTS ####
                        tabPanel("Delta plots",
                                 h3("Delta plot"),
                                 p('Delta plot (Angoff & Ford, 1973) compares the proportions of correct answers per
                                   item in the two groups. It displays non-linear transformation of these proportions using
                                   quantiles of standard normal distributions (so called delta scores) for each item for the two
                                   genders in a scatterplot called diagonal plot or delta plot (see Figure). Item is under
                                   suspicion of DIF if the delta point considerably departs from the diagonal. The detection
                                   threshold is either fixed to value 1.5 or based on bivariate normal approximation (Magis &
                                   Facon, 2012).'),

                                 radioButtons('type_threshold', 'Threshold',
                                              list("Fixed", "Normal")
                                 ),
                                 checkboxInput('puri_DP', 'Item purification', FALSE),
                                 conditionalPanel(
                                   condition = "input.puri_DP",
                                   selectInput("puri_DP_type", "Purification method",
                                             c("IPP1" = "IPP1",
                                               "IPP2" = "IPP2",
                                               "IPP3" = "IPP3"
                                               ),
                                             selected = "IPP1")),
                                 plotOutput('deltaplot'),
                                 downloadButton("DP_deltaplot", label = "Download figure"),
                                 br(),
                                 br(),
                                 verbatimTextOutput("dp_text_normal"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(deltaPlotR)'),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Delta scores with fixed threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = 1.5)'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)'),
                                     br(),
                                     br(),
                                     code('# Delta scores with normal threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = "norm", purify = F)'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)')),
                                 br()
                                 ),
                        # * MANTEL-HAENSZEL ####
                        tabPanel("Mantel-Haenszel",
                                 tabsetPanel(
                                   # Summary
                                   tabPanel("Summary",
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            h4('Summary table'),
                                            p('Here you can select ', strong('correction method'), 'for multiple comparison or ',
                                              strong('item purification.')),
                                            selectInput("correction_method_MZ_print", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"
                                                        ),
                                                        selected = "none"),
                                            checkboxInput('puri_MH', 'Item purification', FALSE),
                                            verbatimTextOutput("print_DIF_MH"),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Mantel-Haenszel test'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   tabPanel('Items',
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            h4('Contingency tables and odds ratio calculation'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("difMHSlider_item",
                                                                      "Item",
                                                                      animate = TRUE,
                                                                      min = 1,
                                                                      max = 10,
                                                                      value = 1,
                                                                      step = 1)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("difMHSlider_score",
                                                                      "Cut-score",
                                                                      min = 0,
                                                                      max = 10,
                                                                      value = 1,
                                                                      step = 1))),
                                            fluidRow(column(12, align = "center", tableOutput('table_DIF_MH'))),
                                            uiOutput('ORcalculation'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Contingency table for item 1 and score 12'),
                                                br(),
                                                code('df <- data.frame(data[, 1], group)'),
                                                br(),
                                                code('colnames(df) <- c("Answer", "Group")'),
                                                br(),
                                                code('df$Answer <- relevel(factor(df$Answer, labels = c("Incorrect", "Correct")), "Correct")'),
                                                br(),
                                                code('df$Group <- factor(df$Group, labels = c("Reference Group", "Focal Group"))'),
                                                br(),
                                                code('score <- apply(data, 1, sum)'),
                                                br(),

                                                code('df <- df[score == 12, ]'),
                                                br(),

                                                code('tab <- dcast(data.frame(xtabs(~ Group + Answer, data = df)),
                                                     Group ~ Answer,
                                                     value.var = "Freq",
                                                     margins = T,
                                                     fun = sum)'),
                                                br(),
                                                code('tab'),
                                                br(),

                                                code('# Mantel-Haenszel estimate of OR'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit$alphaMH')),
                                            br()
                                            )
                        )
                        ),
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            h4("Summary table"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_logistic',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_LR',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_logistic'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_logistic',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_LR_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("diflogSlider", "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to total score.
                                              Their size is determined by count of respondents who achieved given level of
                                              total score with respect to the group membership.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            plotOutput('plot_DIF_logistic'),
                                            downloadButton("DP_plot_DIF_logistic", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = F,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$logitPar')),
                                            br()
                                            )
                                   )
                                 ),

                        # * LOGISTIC Z ####
                        tabPanel("Logistic IRT Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) = \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}{1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4('Summary table'),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison.'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_logistic_IRT_Z',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logZSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"))),
                                            verbatimTextOutput('print_DIF_logistic_IRT_Z'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison.'),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_logistic_IRT_Z',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_logZItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none")),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput("diflog_irtSlider", "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score with respect to the group membership.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            plotOutput('plot_DIF_logistic_IRT_Z'),
                                            downloadButton("DP_plot_DIF_logistic_IRT_Z", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic_IRT_Z'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "none",
                                                     purify = F)'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = T,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('# Coefficients for item 1 - recalculation'),
                                                br(),
                                                code('coef_old <- fit$logitPar[1, ]'),
                                                br(),
                                                code('coef <- c()'),
                                                br(),
                                                code('# a = b1, b = -b0/b1, adif = b3, bdif = -(b1b2-b0b3)/(b1(b1+b3))'),
                                                br(),
                                                code('coef[1] <- coef_old[2]'),
                                                br(),
                                                code('coef[2] <- -(coef_old[1] / coef_old[2])'),
                                                br(),
                                                code('coef[3] <- coef_old[4]'),
                                                br(),
                                                code('coef[4] <- -(coef_old[2] * coef_old[3] + coef_old[1] * coef_old[4] ) /
                                                     (coef_old[2] * (coef_old[2] + coef_old[4]))')),
                                            br()
                                                )
                                            )
                                   ),
                        # * NONLINEAR Z ####
                        tabPanel("Nonlinear IRT Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Nonlinear regression on standardized total scores with IRT parameterization'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c '),
                                              a('(Drabinova & Martinkova, 2017). ',
                                                href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                                                target = "_blank"),
                                              'Similarly to logistic regression, also nonlinear regression allows for
                                              detection of uniform and non-uniform DIF by adding a group specific intercept',
                                              strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'),
                                              '(non-uniform DIF) into the model and by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Summary table"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DIF_NLR',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_nlrSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_NLR_print',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_NLR'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "none")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Nonlinear regression on standardized total scores with IRT parameterization'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c '),
                                              a('(Drabinova & Martinkova, 2017). ',
                                                href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                                                target = "_blank"),
                                              'Similarly to logistic regression, also nonlinear regression allows
                                              for detection of uniform and non-uniform DIF by adding a group specific intercept', strong('bDIF'), '(uniform DIF) and group specific
                                              interaction', strong('aDIF'), '(non-uniform DIF) into the model and by testing for
                                              their significance.'),
                                            h4("Plot with estimated DIF nonlinear curve"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_NLR',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_nlrItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_NLR_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difnlrSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score with respect to the group membership.'),
                                            plotOutput('plot_DIF_NLR'),
                                            downloadButton("DP_plot_DIF_NLR", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_NLR'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "none")'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plot(fit, item = 1)'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$nlrPAR')),
                                            br()
                                            )
                                            )
                        ),
                        # * IRT LORD ####
                        tabPanel("IRT Lord",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            img(src = "fig_lord_uniform.png",
                                                style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                                            img(src = "fig_lord_nonuniform.png",
                                                style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_print_DIF_IRT_lord',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_lordSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput('puri_Lord', 'Item purification', FALSE))),
                                            verbatimTextOutput('print_DIF_IRT_Lord'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_IRT_lord',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_lordItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput('puri_Lord_plot', 'Item purification', FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difirt_lord_itemSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            plotOutput('plot_DIF_IRT_Lord'),
                                            downloadButton("DP_plot_DIF_IRT_Lord", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_lord'),
                                            uiOutput('irteq_lord'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Lord'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef1PL <- fit1PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Lord")'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef2PL <- fit2PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Lord")'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef3PL <- fit3PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Lord")')),
                                            br()
                                            )
                                   )
                                   ),
                        # * IRT RAJU ####
                        tabPanel("IRT Raju",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            img(src = "fig_raju_uniform.png",
                                                style = "float: left; width: 32%; margin-right: 2%; margin-left: 16%; margin-bottom: 0.5em;"),
                                            img(src = "fig_raju_nonuniform.png",
                                                style = "float: left; width: 32%; margin-right: 16%; margin-left: 2%; margin-bottom: 0.5em;"),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_print_DIF_IRT_raju',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_rajuSummary",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_Raju',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DIF_IRT_Raju'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('Here you can choose ', strong('model'), ' to test DIF. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 10%; ",
                                                          radioButtons(inputId = 'type_plot_DIF_IRT_raju',
                                                                       label = 'Model',
                                                                       choices = c("1PL" = '1PL',
                                                                                   "2PL" = '2PL',
                                                                                   "3PL" = '3PL'),
                                                                       selected = '2PL')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_DIF_IRT_rajuItems",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_Raju_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "difirt_raju_itemSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            plotOutput('plot_DIF_IRT_Raju'),
                                            downloadButton("DP_plot_DIF_IRT_Raju", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_raju'),
                                            uiOutput('irteq_raju'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Raju'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 1PL IRT MODEL'),
                                                br(),
                                                code('fit1PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "1PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit1PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef1PL <- fit1PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef1PL, item = 1, test = "Raju")'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit2PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit2PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef2PL <- fit2PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef2PL, item = 1, test = "Raju")'),
                                                br(),
                                                br(),
                                                code('# 3PL IRT MODEL with the same guessing for groups'),
                                                br(),
                                                code('guess <- itemParEst(data, model = "3PL")[, 3]'),
                                                br(),
                                                code('fit3PL <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "3PL", c = guess,
                                                     p.adjust.method = "none", purify = F)'),
                                                br(),
                                                code('fit3PL'),
                                                br(),
                                                code('# Coefficients for all items'),
                                                br(),
                                                code('tab_coef3PL <- fit3PL$itemParInit'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef3PL, item = 1, test = "Raju")')),
                                            br())
                                   )
                                   ),
                        # * DDF ####
                        tabPanel("DDF",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4('Equation'),
                                            p('For ', strong('K'), ' possible test choices is the probability of the correct answer for
                                               person ', strong('i'), ' with standardized total score ', strong('Z'), ' and group
                                              membership ', strong('G'),' in item ', strong('j'), 'given by the following equation: '),
                                            ('$$\\mathrm{P}(Y_{ij} = K|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{1}{1 + \\sum_l e^{\\left( b_{il0} + b_{il1} Z + b_{il2} G + b_{il3} Z:G\\right)}}$$'),
                                            p('The probability of choosing distractor ', strong('k'), ' is then given by: '),
                                            ('$$\\mathrm{P}(Y_{ij} = k|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{e^{\\left( b_{jk0} + b_{jk1} Z_i + b_{jk2} G_i + b_{jk3} Z_i:G_i\\right)}}
                                                   {1 + \\sum_l e^{\\left( b_{jl0} + b_{jl1} Z_i + b_{jl2} G_i + b_{jl3} Z_i:G_i\\right)}}$$'),
                                            br(),
                                            h4('Summary table'),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_print_DDF',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_print_DDF",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_DDF_print',
                                                                        label = 'Item purification',
                                                                        value = FALSE))),
                                            verbatimTextOutput('print_DDF'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "none")'),
                                                br(),
                                                code('fit')),
                                            br()
                                   ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4("Plot with estimated DDF curves"),
                                            p('Here you can choose what', strong('type'), 'of DIF to test. You can also select ',
                                              strong('correction method'), 'for multiple comparison or ', strong('item purification. ')),
                                            fluidPage(div(style = "display: inline-block; vertical-align: top; width: 27%; ",
                                                          radioButtons(inputId = 'type_plot_DDF',
                                                                       label = 'Type',
                                                                       choices = c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                                                   "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                                                   "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'),
                                                                       selected = 'both')),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          selectInput(inputId = "correction_method_plot_DDF",
                                                                      label = "Correction method",
                                                                      choices = c("BH" = "BH",
                                                                                  "Holm" = "holm",
                                                                                  "Hochberg" = "hochberg",
                                                                                  "Hommel" = "hommel",
                                                                                  "BY" = "BY",
                                                                                  "FDR" = "fdr",
                                                                                  "none" = "none"),
                                                                      selected = "none"),
                                                          checkboxInput(inputId = 'puri_DDF_plot',
                                                                        label = 'Item purification',
                                                                        value = FALSE)),
                                                      div(style = "display: inline-block; vertical-align: top; width: 5%; "),
                                                      div(style = "display: inline-block; vertical-align: top; width: 20%; ",
                                                          sliderInput(inputId = "ddfSlider",
                                                                      label = "Item",
                                                                      min = 1,
                                                                      value = 1,
                                                                      max = 10,
                                                                      step = 1,
                                                                      animate = TRUE))),
                                            p('Points represent proportion of selected answer with respect to standardized
                                              total score. Their size is determined by count of respondents who achieved
                                              given level of standardized total score and who selected given option with
                                              respect to the group membership.'),
                                            plotOutput('plot_DDF'),
                                            downloadButton("DP_plot_DDF", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('DDFeq'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DDF'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "none")'),
                                                br(),
                                                code('# Estimated coefficients of item 1'),
                                                br(),
                                                code('fit$mlrPAR[[1]]')),
                                            br()
                                            )
                                 )
                                 )
                        ),
                #%%%%%%%%%%%%%%%%%%%%%
                # REPORTS ############
                #%%%%%%%%%%%%%%%%%%%%%
                tabPanel("Reports",
                         h3("Download report"),
                         h4("Settings of report"),
                         p(code("ShinyItemAnalysis"), " offers an option to download a report in HTML or PDF format. PDF report
                           creation requires latest version of", a("MiKTeX", href = "https://miktex.org/howto/install-miktex",
                                                                   target = "_blank"),
                           "(or other TeX distribution). If you don't have the latest installation, please, use the HTML report."),
                         p("There is an option whether to use customize settings. By checking the", strong("Customize settings"),
                           "local settings will be offered and use for each selected section of report. Otherwise the settings
                           will be taken from pages of application. You can also include your name into report as well as the name
                           of dataset which was used. "),
                         fluidRow(
                           column(2, radioButtons("report_format", "Format of report", c("HTML" = "html", "PDF" = "pdf"))),
                           column(2, checkboxInput("customizeCheck", "Customize settings", FALSE)),
                           column(2, textInput("reportAuthor", "Author")),
                           column(2, textInput("reportDataName", "Dataset"))
                         ),
                         h4("Content of report"),
                         p("Reports by default contain summary of total scores, table of standard scores, item analysis,
                           distractors plots for each item and multinomial regression plots for each item. Other analyses
                           can be selected below. "),
                         fluidRow(
                           column(8,
                                  p(strong("Validity")),
                                  checkboxInput("corr_report", "Correlation structure", FALSE),
                                  conditionalPanel(condition = "input.customizeCheck",
                                                   conditionalPanel(condition = "input.corr_report",
                                                                    div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                                        numericInput('corr_plot_clust_report',
                                                                                     label = 'Number of clusters',
                                                                                     value = 1,
                                                                                     min = 1,
                                                                                     max = 1)),
                                                                    div(style = "display: inline-block; vertical-align: top; width: 20%;",
                                                                        selectInput('corr_plot_clustmethod_report',
                                                                                    label = 'Clustering method',
                                                                                    choices = list("None" = "none",
                                                                                                   "Ward's"  = "ward.D",
                                                                                                   "Ward's n. 2" = "ward.D2",
                                                                                                   "Single" = "single",
                                                                                                   "Complete" = "complete",
                                                                                                   "Average" = "average",
                                                                                                   "McQuitty" = "mcquitty",
                                                                                                   "Median" = "median",
                                                                                                   "Centroid" = "centroid"))))),
                                  checkboxInput("predict_report", "Predictive validity", FALSE)
                           )
                         ),
                         fluidRow(
                           conditionalPanel(condition = "input.customizeCheck",
                                            column(6,
                                                   p(strong("Difficulty/discrimination plot")),
                                                   splitLayout(sliderInput('DDplotNumGroupsSlider_report','Number of groups:',
                                                                           min   = 1,
                                                                           max   = 5,
                                                                           value = 3),
                                                               sliderInput("DDplotRangeSlider_report", "Which two groups to compare:",
                                                                           min = 1,
                                                                           max = 3,
                                                                           step = 1,
                                                                           value = c(1, 3)))))
                         ),
                         fluidRow(
                           conditionalPanel(condition = "input.customizeCheck",
                                            column(6,
                                                   p(strong("Distractors plots")),
                                                   splitLayout(radioButtons('type_combinations_distractor_report',
                                                                            'Type',
                                                                            list("Combinations", "Distractors")),
                                                               sliderInput('distractorGroupSlider','Number of groups:',
                                                                             min   = 1,
                                                                             max   = 5,
                                                                             value = 3))))
                         ),
                         fluidRow(
                           column(4,
                                  radioButtons("irt_type_report", "IRT model selection",
                                               c("None" = "none",
                                                 "Rasch" = "rasch",
                                                 "1PL" = "1pl",
                                                 "2PL" = "2pl",
                                                 "3PL" = "3pl",
                                                 "4PL" = "4pl"),
                                               selected = "1pl")
                           )
                         ),

                           fluidRow(
                             column(3,
                                    p(strong("DIF method selection")),
                                    checkboxInput("histCheck", "None - histograms by group only", FALSE),
                                    checkboxInput("deltaplotCheck", "Delta plot", FALSE),
                                    checkboxInput("logregCheck", "Logistic regression", FALSE),
                                    checkboxInput("multiCheck", "Multinomial regression", FALSE)
                             ),
                             conditionalPanel(condition = "input.customizeCheck",
                             conditionalPanel(condition = "input.deltaplotCheck",
                               column(2, p(strong("Delta plot settings")),
                                      radioButtons('type_threshold_report', 'Threshold',
                                                   list("Fixed", "Normal")
                                      ),
                                      checkboxInput('puri_DP_report', 'Item purification', FALSE),
                                      conditionalPanel(
                                        condition = "input.puri_DP_report",
                                        selectInput("puri_DP_type_report", "Purification method",
                                                    c("IPP1" = "IPP1",
                                                      "IPP2" = "IPP2",
                                                      "IPP3" = "IPP3"
                                                    ),
                                                    selected = "IPP1")
                                      )
                               )
                             ),
                             conditionalPanel(condition = "input.logregCheck",
                               column(2, p(strong("Logistic regression settings")),
                                      radioButtons('type_print_DIF_logistic_report', 'Type',
                                                   c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                     "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                     "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                   ),
                                                   'both'
                                      ),
                                      selectInput("correction_method_log_report", "Correction method",
                                                  c("BH" = "BH",
                                                    "Holm" = "holm",
                                                    "Hochberg" = "hochberg",
                                                    "Hommel" = "hommel",
                                                    "BY" = "BY",
                                                    "FDR" = "fdr",
                                                    "none" = "none"
                                                  ),
                                                  selected = "none"),
                                      checkboxInput('puri_LR_report', 'Item purification', FALSE)
                               )
                             ),
                             conditionalPanel(condition = "input.multiCheck",
                               column(2, p(strong("Multinomial regression settings")),
                                      radioButtons('type_DDF_report', 'Type',
                                                   c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                     "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                     "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                   ),
                                                   'both'
                                      ),
                                      selectInput("correction_method_DDF_report", "Correction method",
                                                  c("BH" = "BH",
                                                    "Holm" = "holm",
                                                    "Hochberg" = "hochberg",
                                                    "Hommel" = "hommel",
                                                    "BY" = "BY",
                                                    "FDR" = "fdr",
                                                    "none" = "none"),
                                                  selected = "none"),
                                      checkboxInput('puri_DDF_report', 'Item purification', FALSE)
                               )
                            )
                           )
                         ),
                         p(strong("Recommendation: "), "Report generation can be faster and more reliable when you first check
                           sections of intended contents. For example, if you wish to include a ", strong("3PL IRT"),
                           " model, you can first visit ", strong("IRT models"), "section and ", strong("3PL"), " subsection."),
                         #p(strong("Warning: "), "Download of reports takes some time. Please, be patient."),
                         fluidRow(
                           column(width = 5,
                             splitLayout(cellWidths = c("45%", "55%"),
                               actionButton("generate", "Generate report"),
                               uiOutput("download_report_button")
                             )
                           )
                         ),
                         br(),
                         br(),
                         br()
                         ),

             #%%%%%%%%%%%%%%%%%%%%%
             # REFERENCES #########
             #%%%%%%%%%%%%%%%%%%%%%
             tabPanel("References",
                      h3('References'),
                      p('Akaike, H. (1974). A New Look at the Statistical Model Identification. IEEE Transactions
                        on Automatic Control, 19(6), 716-723.', a('See online.',
                                                                  href = "http://ieeexplore.ieee.org/abstract/document/1100705/",
                                                                  target = "_blank")),
                      p('Ames, A. J., & Penfield, R. D. (2015). An NCME Instructional Module on Item-Fit
                        Statistics for Item Response Theory Models. Educational Measurement: Issues and
                        Practice, 34(3), 39-48.', a("See online.",
                                                    href = "http://onlinelibrary.wiley.com/doi/10.1111/emip.12067/full",
                                                    target = "_blank")),
                      p('Andrich, D. (1978). A rating formulation for ordered response categories.
                        Psychometrika, 43(4), 561-573.',
                        a("See online.",
                          href = "https://link.springer.com/article/10.1007/BF02293814",
                          target = "_blank")),
                      p('Angoff, W. H., & Ford, S. F. (1973). Item-Race Interaction on a Test of
                        Scholastic Aptitude. Journal of Educational Measurement, 10(2), 95-105.',
                        a("See online.",
                          href = "https://www.jstor.org/stable/1433905?seq=1#page_scan_tab_contents",
                          target = "_blank")),
                      p('Bock, R. D. (1972). Estimating Item Parameters and Latent Ability when
                        Responses Are Scored in Two or More Nominal Categories. Psychometrika,
                        37(1), 29-51.', a("See online.",
                                          href = "http://link.springer.com/article/10.1007/BF02291411",
                                          target = "_blank")),
                      p('Cronbach, L. J. (1951). Coefficient Alpha and the Internal Structure
                        of Tests. Psychometrika, 16(3), 297-334.', a("See online.",
                                                                     href = "https://link.springer.com/article/10.1007/BF02310555",
                                                                     target = "_blank")),
                      p("Drabinova, A., & Martinkova, P. (2017). Detection of Differential Item Functioning with Non-Linear
                         Regression: Non-IRT Approach Accounting for Guessing. Journal of Educational Measurement, 54(4), 498-517.",
                        a("See online.",
                          href = "http://onlinelibrary.wiley.com/doi/10.1111/jedm.12158/full",
                          target = "_blank")),
                      p("Lord, F. M. (1980). Applications of Item Response Theory to Practical Testing Problems.
                        Routledge."),
                      p("Magis, D., & Facon, B. (2012). Angoff's Delta Method Revisited:
                        Improving DIF Detection under Small Samples. British Journal of
                        Mathematical and Statistical Psychology, 65(2), 302-321.", a("See online.",
                                                                                     href = "https://www.ncbi.nlm.nih.gov/pubmed/22500570",
                                                                                     target = "_blank")),
                      p("Mantel, N., & Haenszel, W. (1959). Statistical Aspects of the Analysis of Data from
                        Retrospective Studies. Journal of the National Cancer Institute, 22 (4), 719-748.", a("See online.",
                                                                                                              href = "http://www.medicine.mcgill.ca/epidemiology/hanley/c634/stratified/Mantel_Haenszel_1.pdf",
                                                                                                              target = "_blank")),
                      p("Martinkova, P., Drabinova, A., & Houdek, J. (2017). ShinyItemAnalysis: Analyza prijimacich a
                         jinych znalostnich ci psychologických testu. TESTFORUM, 6(9), 16–35.",
                        a("See online.", href = "http://testforum.cz/domains/testforum.cz/index.php/testforum/article/view/TF2017-9-129", target = "_blank"),
                        "(ShinyItemAnalysis: Analyzing admission and other educational and psychological tests)"),
                      p("Martinkova, P., Drabinova, A., Liaw, Y. L., Sanders, E. A., McFarland, J. L., & Price, R. M.
                        (2017). Checking Equity: Why Differential Item Functioning Analysis Should Be a Routine Part
                        of Developing Conceptual Assessments. CBE-Life Sciences Education, 16(2). ",
                        a('See online.',
                          href = "https://www.lifescied.org/doi/10.1187/cbe.16-10-0307",
                          target = "_blank")),
                      p("Martinkova, P., Stepanek, L., Drabinova, A., Houdek, J., Vejrazka, M., & Stuka, C. (2017).
                        Semi-real-time analyses of item characteristics for medical school admission tests. In: Proceedings of
                        the 2017 Federated Conference on Computer Science and Information Systems. In print."),
                      p("Masters, G. N. (1982). A Rasch model for partial credit scoring. Psychometrika, 47(2), 149-174.",
                        a('See online.',
                          href = "https://link.springer.com/article/10.1007/BF02296272",
                          target = "_blank")),
                      p("McFarland, J. L., Price, R. M., Wenderoth, M. P., Martinkova, P., Cliff, W., Michael, J., ... & Wright, A. (2017).
                        Development and validation of the homeostasis concept inventory.
                        CBE-Life Sciences Education, 16(2), ar35.",
                        a('See online.',
                          href = "https://www.lifescied.org/doi/abs/10.1187/cbe.16-10-0305",
                          target = "_blank")),
                      p("Muraki, E. (1992). A generalized partial credit model: Application of an EM algorithm.
                        ETS Research Report Series, 1992(1).",
                        a('See online.',
                          href = "https://onlinelibrary.wiley.com/doi/abs/10.1002/j.2333-8504.1992.tb01436.x",
                          target = "_blank")),
                      p("Swaminathan, H., & Rogers, H. J. (1990). Detecting Differential Item
                        Functioning Using Logistic Regression Procedures. Journal of Educational
                        Measurement, 27(4), 361-370.",
                        a('See online.',
                          href = "https://www.jstor.org/stable/1434855?seq=1#page_scan_tab_contents",
                          target = "_blank")),
                      p("Raju, N. S. (1988). The Area between Two Item Characteristic Curves. Psychometrika,
                        53 (4), 495-502.",
                        a('See online.',
                          href = "https://link.springer.com/article/10.1007/BF02294403",
                          target = "_blank")),
                      p("Raju, N. S. (1990). Determining the Significance of Estimated Signed and Unsigned Areas
                        between Two Item Response Functions. Applied Psychological Measurement, 14 (2), 197-207.",
                        a('See online.',
                          href = "http://journals.sagepub.com/doi/abs/10.1177/014662169001400208",
                          target = "_blank")),
                      p('Rasch, G. (1960) Probabilistic Models for Some Intelligence and Attainment Tests.
                        Copenhagen: Paedagogiske Institute.'),
                      p("Samejima, F. (1969). Estimation of latent ability using a response pattern of graded scores.
                        Psychometrika, 34(1), 1-97.",
                        a('See online.',
                          href = "https://link.springer.com/article/10.1007%2FBF03372160",
                          target = "_blank")),
                      p('Schwarz, G. (1978). Estimating the Dimension of a Model. The Annals of Statistics,
                        6(2), 461-464.', a('See online.',
                                           href = "https://projecteuclid.org/euclid.aos/1176344136",
                                           target = "_blank")),
                      p("Wilson, M. (2005). Constructing Measures: An Item Response Modeling Approach."),
                      p("Wright, B. D., & Stone, M. H. (1979). Best Test Design. Chicago: Mesa Press."),
                      br()
                      )
                      )
                      )

