#' Entropy rate per time
#' 
#' Computes the entropy rate per time of the Markov chain generated by the compartmental system
#' 
#' @param A A constant compartmental square matrix with cycling rates in the
#' diagonal and transfer rates in the off-diagonal.
#' @param u A one-column matrix defining the amount of inputs per compartment.
#' @return A scalar value with the entropy rate per time
#' @references 
#' Metzler, H. (2020). Compartmental systems as Markov chains : age, 
#' transit time, and entropy (T. Oertel-Jaeger, I. Pavlyukevich, and C. Sierra, 
#' Eds.) 
#' [PhD thesis](https://suche.thulb.uni-jena.de/Record/1726091651)
#' @examples
#' B6=matrix(c(-1,1,0,0,-1,1,0,0,-1),3,3); u6=matrix(c(1,0,0))
#' entropyRatePerTime(A=B6, u=u6)
entropyRatePerTime=function(A,u){
  x=-1*solve(A)%*%u
  ET=sum(x)/sum(u)
  Hp=pathEntropy(A,u)
  thetaZ=Hp/ET
  return(thetaZ)
}
