% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation-spades.R
\name{spades}
\alias{spades}
\alias{spades,simList,ANY,ANY,missing-method}
\alias{spades,ANY,ANY,ANY,logical-method}
\title{Run a spatial discrete event simulation}
\usage{
spades(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  events = NULL,
  .plots = getOption("spades.plots", NULL),
  ...
)

\S4method{spades}{simList,ANY,ANY,missing}(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  events = NULL,
  .plots = getOption("spades.plots", NULL),
  ...
)

\S4method{spades}{ANY,ANY,ANY,logical}(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  events = NULL,
  .plots = getOption("spades.plots", NULL),
  ...
)
}
\arguments{
\item{sim}{A \code{simList} simulation object, generally produced by \code{simInit}.}

\item{debug}{Optional tools for invoking debugging. Supplying a \code{list}
will invoke the more powerful \code{logging} package. See details.
Default is to use the value in \code{getOption("spades.debug")}.}

\item{progress}{Logical (\code{TRUE} or \code{FALSE} show a graphical progress bar),
character (\code{"graphical"}, \code{"text"}) or numeric indicating
the number of update intervals to show in a graphical progress bar.}

\item{cache}{Logical. If \code{TRUE}, then the \code{spades} call will be cached.
This means that if the call is made again with the same \code{simList},
then \code{spades} will return the return value from the previous run
of that exact same \code{simList}. Default \code{FALSE}. See Details.
See also the vignette on caching for examples.}

\item{.plotInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{.saveInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{notOlderThan}{Date or time. Passed to \code{reproducible::Cache} to update the cache.
Default is \code{NULL}, meaning don't update the cache.
If \code{Sys.time()} is provided, then it will force a recache,
i.e., remove old value and replace with new value.
Ignored if \code{cache} is \code{FALSE}.}

\item{events}{A character vector or a named list of character vectors. If specified,
the simulations will only do the events indicated here. If a named list, the names
must correspond to the modules and the character vectors can be specific events within
each of the named modules. With the \code{list} form, all unspecified modules
will run \emph{all} their events, including internal spades modules, e.g., \code{save},
that get invoked with the \code{outputs} argument in  \code{simInit}. See example.}

\item{.plots}{Character. Sets the parameter of this name in all modules.
See \code{\link[=Plots]{Plots()}} for possible values. The parameter is intended to slowly
take over from \code{.plotInitialTime} as a mechanism to turn on or off plotting.
For backwards compatibility, if \code{.plotInitialTime} is not set in this \code{spades} call,
but this \code{.plots} is used, two things will happen: setting this without \code{"screen"}
will turn off all plotting; setting this with \code{"screen"} will trigger
plotting for any modules that use this parameter but will have no effect on
other modules. To get plotting, therefore, it may be necessary to also set
\code{.plotInitialTime = start(sim)}.}

\item{...}{Any. Can be used to make a unique cache identity, such as "replicate = 1".
This will be included in the \code{Cache} call, so will be unique
and thus \code{spades} will not use a cached copy as long as
anything passed in \code{...} is unique, i.e., not cached previously.}
}
\value{
Invisibly returns the modified \code{simList} object.
}
\description{
Here, we implement a simulation in a more modular fashion so it's easier to add
submodules to the simulation. We use S4 classes and methods, and use \code{data.table}
instead of \code{data.frame} to implement the event queue (because it is much faster).
}
\details{
The is the workhorse function in the SpaDES package. It runs simulations by
implementing the rules outlined in the \code{simList}.

This function gives simple access to two sets of module parameters:
\code{.plotInitialTime} and with \code{.plotInitialTime}. The primary use of
these arguments is to temporarily turn off plotting and saving. "Temporary"
means that the \code{simList} is not changed, so it can be used again with
the \code{simList} values reinstated. To turn off plotting and saving, use
\code{.plotInitialTime = NA} or \code{.saveInitialTime = NA}. NOTE: if a
module did not use \code{.plotInitialTime} or \code{.saveInitialTime}, then
these arguments will not do anything.
}
\note{
The debug option is primarily intended to facilitate building simulation
models by the user.
Will print additional outputs informing the user of updates to the values of
various \code{simList} slot components.
See \url{https://github.com/PredictiveEcology/SpaDES/wiki/Debugging} for details.
}
\section{Caching with SpaDES}{


There are numerous ways in which Caching can be used within SpaDES. Please
see the vignette
\url{https://spades-core.predictiveecology.org/articles/iii-cache.html}
for many examples. Briefly, functions, events, modules, entire spades calls or
experiment calls (see \url{https://github.com/PredictiveEcology/SpaDES.experiment})
can be cached and mixtures of all of these will work. For functions, simply
wrap the call with \code{Cache}, moving the original function name into
the first argument of Cache. For events or modules, set the module \code{parameters},
\code{.useCache}, e.g.,
\code{simInit(..., parameters = list(myModule = list(.useCache = "init")))}.
This can be set to an event name, which will cache that event, or a logical,
which will cache \emph{every} event in that module. Event and module caching
makes most sense when the event or module only runs once, such as an initialization
or data preparation event/module. Caching an entire simulation is actually just
a function call to \code{simInitAndSpades}, for example. So, simply writing
\code{Cache(simInitAndSpades, modules = ...)} will effectively cache a whole simulation.
Finally for experiments, it is just like a function call:
\code{Cache(simInitandExperiment, ...)}. The final way Caching can be done is in
\code{experiment} or \code{spades}, by setting the \code{cache} argument.

If \code{cache} is TRUE, this allows for a seamless way to "save" results
of a simulation. The  user does not have to intentionally do any saving manually.
Instead, upon a call to \code{spades} in which the \code{simList} is identical,
the function will simply return the result that would have come if it had
been rerun. Use this with caution, as it will return exactly the result from
a previous run, even if there is stochasticity internally.
Caching is only based on the input \code{simList.}
See also the vignette on caching for examples.
}

\section{\code{debug}}{


The most powerful way to use debug is to invoke the \code{logging}
R package. To invoke this, \code{debug} must be a list with up to 3
named elements:
\code{console}, \code{file}, and \code{debug}. Each of these list elements
must be a list (including empty \code{list()} for defaults) with the
sub-list elements here:
\tabular{lll}{
\code{console} \tab \code{level} \tab The \code{level}, see below, of information shown\cr
\code{file} \tab \code{append} \tab Logical. If \code{TRUE}, the default, then
log entries are appended to file, if it exists\cr
\tab \code{file} \tab A filename. Defaults to \code{log.txt}\cr
\tab \code{level} \tab The \code{level}, see below, of information shown\cr
\code{debug} \tab See possible values below\cr
}

\code{level} can be a number from 0 to 100 or a character string matching one
of the values in \code{logging::loglevels}. These are hierarchical levels of
information passed to the console. Set a lower number for more information and a
higher number for less information. Errors in code will be shown if \code{level}
is set to \code{"ERROR"} or \code{40} or above; warnings in code will be shown if
\code{level} is set to \code{"WARN"} or \code{30} or above;
normal messages in code will
be shown if \code{level} is set to \code{"INFO"} or \code{20} or above. For
consistency with base R messaging, if default level is used, then normal
messaging via \code{message} will be shown; this means that \code{suppressMessages}
will work to suppress messaging only when level is set to \code{"INFO"} or \code{20}.
Some functions in the SpaDES ecosystem may have information at the lower levels,
but currently, there are few to none.

\code{debug} is specified as a non-list argument to \code{spades} or as
\code{list(debug = ...)}, then it can be a logical, a quoted call, a character vector
or a numeric scalar (currently 1 or 2) or a list of any of these to get multiple
outputs. This will be run at the start of every event. The following options for debug
are available. Each of these can also be in a list to get multiple outputs:

\tabular{ll}{
\code{TRUE} \tab \code{current(sim)} will be printed at the start of each event as
it runs\cr
a function name (as character string) \tab If a function, then it will be run on the
\code{simList}, e.g., "time" will run
\code{time(sim)} at each event.\cr
\code{moduleName} (as character string) \tab All calls to that module will be entered
interactively\cr
\code{eventName} (as character string) \tab All calls that have that event name (in any module)
will be entered interactively\cr
\verb{c(<moduleName>, <eventName>)}  \tab Only the event in that specified module
will be entered into. \cr
Any other R expression expressed as a character string or quoted call \tab
Will be evaluated with access to the \code{simList} as \code{sim}.
If this is more than one character string, then all will
be printed to the screen in their sequence. \cr
A numeric scalar, currently 1 or 2 (maybe others) \tab This will print out alternative forms of event
information that users may find useful \cr
}

If not specified in the function call, the package
option \code{spades.debug} is used.

If \code{options("spades.browserOnError" = TRUE)} (experimental still) if
there is an error, it will attempt to open a browser
in the event where the error occurred. You can edit, and then press \code{c} to continue
or \code{Q} to quit, plus all other normal interactive browser tools.
\code{c} will trigger a reparse and events will continue as scheduled, starting
with the one just edited. There may be some unexpected consequences if the
\code{simList} objects had already been changed before the error occurred.
}

\examples{
\donttest{
if (requireNamespace("SpaDES.tools", quietly = TRUE) &&
    requireNamespace("NLMR", quietly = TRUE)) {
  # some options are not necessary when not interactive
  opts <- options("spades.moduleCodeChecks" = FALSE, "spades.useRequire" = FALSE)
  if (!interactive()) opts <- append(opts, options("spades.plots" = NA))
  mySim <- simInit(
   times = list(start = 0.0, end = 1.0, timeunit = "year"),
   params = list(
     randomLandscapes = list(nx = 10, ny = 10),
     .globals = list(stackName = "landscape", burnStats = "nPixelsBurned",
                     .plots = NA) # plotting off --> not relevant for example
   ),
   modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
   paths = list(modulePath = getSampleModules(tempdir()))
  )
  spades(mySim)

  options(opts) # reset options
}
}

}
\references{
Matloff, N. (2011). The Art of R Programming (ch. 7.8.3).
San Francisco, CA: No Starch Press, Inc..
Retrieved from \url{https://nostarch.com/artofr.htm}
}
\seealso{
\code{\link[=SpaDES.core-package]{SpaDES.core-package()}},
\code{\link[=simInit]{simInit()}}, and the caching vignette (very important for reproducibility):
\url{https://spades-core.predictiveecology.org/articles/iii-cache.html} which
uses \code{\link[reproducible:Cache]{reproducible::Cache()}}.

vignettes
}
\author{
Alex Chubaty and Eliot McIntire
}
