% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aba_screen.R
\name{aba_screen}
\alias{aba_screen}
\title{Create an aba screen object.}
\usage{
aba_screen(
  object,
  threshold,
  cost_multiplier,
  include_n,
  ntrials = 100,
  verbose = TRUE
)
}
\arguments{
\item{object}{an aba model. The fitted aba model which you want to use as
the screening algorithm.}

\item{threshold}{double or vector of doubles between 0 and 1. The threshold
represents the percentage of individuals who will be invited to take the
inclusion test. Note that the threshold value is calculated in a relative
manner based on the values in the data population, not based on an absolute
risk value.}

\item{cost_multiplier}{double or vector of doubles. The cost multiplier
represents how much more expensive it is to perform the main inclusion
test versus the screening test. Larger values mean that the main inclusion
test/biomarker is much more expensive than the screening test and will
therefore result in larger cost savings by using the screening model to
identify individuals who are at low risk to be positive on the main
inclusion test.}

\item{include_n}{integer. The number of participants who you expect
to be included in the clinical trial. This is therefore the number of
individuals who must pass the screening test and who then must pass the
main inclusion test}

\item{ntrials}{integer. The number of bootstrap trials to run in order to
generate the confidence interval}

\item{verbose}{logical. Whether to show a progress bar for each trial.}
}
\value{
an abaScreen object
}
\description{
This function runs a clinical trial screening analysis based on a fitted aba
model with glm stats. You can supply different inclusion thresholds which
represent predicted probabilities from the glm stats, and you can also
supply cost multipliers and the required sample size in order to perform a
cost-benefit analysis. This analysis uses bootstrap sampling to generate
confidence intervals.
}
\examples{

# use built-in data
df <- adnimerge \%>\% dplyr::filter(VISCODE == 'bl')

# first, fit an aba model to predict amyloid PET status from plasma markers
# In this scenario, PET is the "inclusion" marker and plasma is the
# "screening" marker. PET is expensive and plasma is cheap, so we want to
# use plasma markers to decide who should undergo PET scans in order to
# minimize the risk of negative (i.e., wasted) PET scans.
model <- df \%>\% aba_model() \%>\%
  set_groups(everyone()) \%>\%
  set_outcomes(PET_ABETA_STATUS_bl) \%>\%
  set_predictors(
    PLASMA_PTAU181_bl,
    PLASMA_NFL_bl,
    c(PLASMA_PTAU181_bl, PLASMA_NFL_bl)
  ) \%>\%
  set_covariates(AGE, GENDER, EDUCATION) \%>\%
  set_stats('glm') \%>\%
  fit()

# summarise the model just to show the plasma biomarkers do in fact
# provide some predictive value for amyloid PET status
model_summary <- model \%>\% aba_summary()

# Run the screening analysis while varying the inclusion threshold from
# 25\% to 75\% (this is the percent of individuals who will be invited for
# the PET scan) and varying the cost multiplier from 4 to 16 (this is how
# much more PET costs compared to plasma) and assuming we want to recruit
# 1000 amyloid PET positive subjects.
model_screen <- model \%>\%
  aba_screen(
    threshold = seq(0.25, 0.75, by = 0.1),
    cost_multiplier = c(4, 8, 16),
    include_n = 1000,
    ntrials = 5,
    verbose = TRUE
  )

}
