% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_param_bor.R
\name{derive_param_bor}
\alias{derive_param_bor}
\title{Adds a Parameter for Best Overall Response (without confirmation)}
\usage{
derive_param_bor(
  dataset,
  dataset_adsl,
  filter_source,
  source_pd = NULL,
  source_datasets = NULL,
  reference_date,
  ref_start_window,
  missing_as_ne = FALSE,
  aval_fun,
  set_values_to,
  subject_keys = get_admiral_option("subject_keys")
)
}
\arguments{
\item{dataset}{The input dataframe from which the Best Overall Response will
be derived from and added to.

The columns \code{PARAMCD}, \code{ADT}, and \code{AVALC}and the columns specified in
\code{subject_keys} and \code{reference_date} are expected.

After applying \code{filter_source} and/or \code{source_pd} the column \code{ADT} and the
columns specified by \code{subject_keys} must be a unique key of the dataframe.

\emph{Permitted Values:} a \code{data.frame()} object}

\item{dataset_adsl}{ADSL input dataset.

The columns specified in the subject_keys argument are expected. For each subject in
the passed \code{dataset} a new row is added to the input \code{dataset}. Columns
in \code{dataset_adsl} that also appear in \code{dataset} will be populated with the
appropriate subject-specific value for these new rows.

\emph{Permitted Values:} a \code{data.frame()} object}

\item{filter_source}{Filter to be applied to \code{dataset} to derive the
Best Overall Response}

\item{source_pd}{Date of first progressive disease (PD)

If the parameter is specified, the observations of the input \code{dataset} for
deriving the new parameter are restricted to observations up to the
specified date. Observations at the specified date are included. For
subjects without first PD date all observations are take into account.

\emph{Permitted Values:} a \code{date_source} object (see \code{date_source()}
for details)}

\item{source_datasets}{Source dataframe to be used to calculate the
first PD date

A named list of dataframes is expected (although for BOR) only one dataframe is
needed. It links the \code{dataset_name} from \code{source_pd} with an existing dataframe.

For example if \code{source_pd = pd_date} with

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pd_date <- date_source(
  dataset_name = "adrs",
  date = ADT,
  filter = PARAMCD == PD
)
}\if{html}{\out{</div>}}

and the actual response dataframe in the script is \code{myadrs}, \code{source_datasets = list(adrs = myadrs)} should be specified.}

\item{reference_date}{Reference date

The reference date is used along with \code{ref_start_window} to determine those
records that occur before and after \code{ADT} (see Details section for further
information). Usually it is treatment start date (\code{TRTSDT}) or
randomization date (\code{RANDDT}).

\emph{Permitted Values:} a numeric date column}

\item{ref_start_window}{Stable disease time window

The ref_start_window is used along with \code{reference_date} to determine those
records that occur before and after \code{ADT} (i.e. for a record determine
whether \code{ADT} >= \code{reference_date} + \code{ref_start_window}),
see Details section for further information.

\emph{Permitted Values:} a non-negative numeric scalar}

\item{missing_as_ne}{Consider no assessments as \code{"NE"}?

If the argument is set to \code{TRUE}, the response is set to \code{"NE"} for
subjects in \code{dataset_adsl} without an assessment in the \code{dataset} after the
filter has been applied. Otherwise, the response is set to \code{"MISSING"}
for these subjects.

\emph{Permitted Values:} a logical scalar}

\item{aval_fun}{\emph{Deprecated}, please use \code{set_values_to} instead.

Function to map character analysis value (\code{AVALC}) to numeric analysis
value (\code{AVAL})

The (first) argument of the function must expect a character vector and the
function must return a numeric vector.}

\item{set_values_to}{New columns to set

A named list returned by \code{exprs()} defining the columns to be set for the
new parameter, e.g. \code{exprs(PARAMCD = "BOR", PARAM = "Best Overall Response")} is expected. The values must be symbols, character strings,
numeric values, or \code{NA}.}

\item{subject_keys}{Columns to uniquely identify a subject

\emph{Permitted Values:} A list of symbols created using \code{exprs()}.}
}
\value{
The dataframe passed in the \code{dataset} argument with additional columns and/or
rows as set in the \code{set_values_to} argument.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#superseded}{\figure{lifecycle-superseded.svg}{options: alt='[Superseded]'}}}{\strong{[Superseded]}} The \code{derive_param_bor()} function has been
superseded in favor of \code{derive_extreme_event()}.

Adds a parameter for best overall response, without confirmation, optionally up to
first progressive disease
}
\details{
Calculates the best overall response (BOR) parameter, as detailed below.

Records after PD can be removed using the source_pd and source_datasets
arguments.

Note:
\enumerate{
\item All \code{CR}, \code{PR} and \code{PD} response records are considered for Best Overall Response.
\item All \code{SD} or \code{NON-CR/NON-PD} records where \code{ADT} >= \code{reference_date} +
\code{ref_start_window} are also considered for Best Overall Response.
\item Subjects with \strong{ONLY} an \code{SD} or \code{NON-CR/NON-PD} records where \code{ADT} <
\code{reference_date} + \code{ref_start_window} are assigned a Best Overall Response of \code{NE}.
\item The Best Response, from the records in steps 1 to 3, is then selected in the following
order of preference: CR, PR, SD, NON-CR/NON-PD, PD, NE, MISSING
\item The \code{AVAL} column is added and set using the \code{aval_fun(AVALC)} function
\item The columns specified by the \code{set_values_to} parameter and records
are added to the dataframe passed into the \code{dataset} argument
}

Note: Any responses of SD or NON-CR/NON-PD that occur before \code{reference_date} +
\code{ref_start_window} are ignored in the calculation of BOR. All other responses are included
in the calculation of BOR, irrespective of the number of days from the reference date.

Also Note: All columns from the input dataset are kept. For subjects with no records in
the input dataset (after the filter is applied) all columns are kept from ADSL which are
also in the input dataset.  Columns which are not to be populated for the new parameter
or populated differently (e.g. \code{RSSTRESC}, \code{VISIT}, \code{PARCATy}, \code{ANLzzFL}, ...) should be
overwritten using the \code{set_values_to} parameter.
}
\examples{

library(magrittr)
library(dplyr)
library(tibble)
library(lubridate)
library(admiral)

# Create ADSL dataset
adsl <- tribble(
  ~USUBJID, ~TRTSDTC,
  "1",      "2020-01-01",
  "2",      "2019-12-12",
  "3",      "2019-11-11",
  "4",      "2019-12-30",
  "5",      "2020-01-01",
  "6",      "2020-02-02",
  "7",      "2020-02-02",
  "8",      "2020-04-01"
) \%>\%
  mutate(
    TRTSDT = ymd(TRTSDTC),
    STUDYID = "XX1234"
  )

# Create ADRS dataset
ovr_obs <- tribble(
  ~USUBJID, ~ADTC, ~AVALC, ~ANL01FL,
  "1", "2020-01-01", "PR", "Y",
  "1", "2020-02-01", "CR", "Y",
  "1", "2020-02-16", "NE", "Y",
  "1", "2020-03-01", "CR", "Y",
  "1", "2020-04-01", "SD", "Y",
  "2", "2020-01-01", "SD", "Y",
  "2", "2020-02-01", "PR", "Y",
  "2", "2020-03-01", "SD", "Y",
  "2", "2020-03-13", "CR", "Y",
  "3", "2019-11-12", "CR", "Y",
  "3", "2019-12-02", "CR", "Y",
  "3", "2020-01-01", "SD", "Y",
  "4", "2020-01-01", "PR", "Y",
  "4", "2020-03-01", "SD", "N",
  "4", "2020-04-01", "SD", "Y",
  "4", "2020-05-01", "PR", "Y",
  "4", "2020-05-15", "NON-CR/NON-PD", "Y",
  "5", "2020-01-01", "PR", "Y",
  "5", "2020-01-10", "SD", "Y",
  "5", "2020-01-20", "PR", "Y",
  "5", "2020-05-15", "NON-CR/NON-PD", "Y",
  "6", "2020-02-06", "PR", "Y",
  "6", "2020-02-16", "CR", "Y",
  "6", "2020-03-30", "PR", "Y",
  "6", "2020-04-12", "PD", "Y",
  "6", "2020-05-01", "CR", "Y",
  "6", "2020-06-01", "CR", "Y",
  "7", "2020-02-06", "PR", "Y",
  "7", "2020-02-16", "CR", "Y",
  "7", "2020-04-01", "NE", "N"
) \%>\%
  mutate(PARAMCD = "OVR")

pd_obs <-
  bind_rows(tribble(
    ~USUBJID, ~ADTC,        ~AVALC,
    "2",      "2020-03-01", "Y",
    "4",      "2020-02-01", "Y"
  ) \%>\%
    mutate(PARAMCD = "PD"))

adrs <- bind_rows(ovr_obs, pd_obs) \%>\%
  mutate(
    ADT = ymd(ADTC),
    STUDYID = "XX1234"
  ) \%>\%
  select(-ADTC) \%>\%
  derive_vars_merged(
    dataset_add = adsl,
    by_vars     = exprs(STUDYID, USUBJID),
    new_vars    = exprs(TRTSDT)
  )

pd_date <- date_source(
  dataset_name = "adrs",
  date         = ADT,
  filter       = PARAMCD == "PD"
)

aval_fun_pass <- function(arg) {
  case_when(
    arg == "CR" ~ 11,
    arg == "PR" ~ 22,
    arg == "SD" ~ 33,
    arg == "NON-CR/NON-PD" ~ 44,
    arg == "PD" ~ 55,
    arg == "NE" ~ 66,
    arg == "MISSING" ~ 77,
    TRUE ~ NA_real_
  )
}

# Derive best overall response parameter
derive_param_bor(
  adrs,
  dataset_adsl = adsl,
  filter_source = PARAMCD == "OVR" & ANL01FL == "Y",
  source_pd = pd_date,
  source_datasets = list(adrs = adrs),
  aval_fun = aval_fun_pass,
  reference_date = TRTSDT,
  ref_start_window = 28,
  set_values_to = exprs(
    PARAMCD = "BOR",
    PARAM = "Best Overall Response"
  )
) \%>\%
  filter(PARAMCD == "BOR")
}
\seealso{
Other superseded: 
\code{\link{derive_param_clinbenefit}()},
\code{\link{derive_param_confirmed_bor}()},
\code{\link{derive_param_confirmed_resp}()},
\code{\link{derive_param_response}()},
\code{\link{filter_pd}()}
}
\author{
Stephen Gormley
}
\concept{superseded}
\keyword{superseded}
