% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/envelope.R
\name{envelope}
\alias{envelope}
\title{Generate Simulated Envelope}
\usage{
envelope(
  model,
  residual_fn = envelope_residual(model),
  alpha = 0.05,
  nsim = 100,
  responses = NULL,
  no_warnings = FALSE,
  no_messages = FALSE,
  converged_only = FALSE,
  show_progress = TRUE,
  plot.it = TRUE,
  refit_fn = NULL,
  ...
)
}
\arguments{
\item{model}{A model to generate responses and compute the observed residuals.}

\item{residual_fn}{A function to calculate model residuals. The default is
\code{\link[=envelope_residual]{envelope_residual()}} for an absolute residual.}

\item{alpha}{The significance level for constructing the envelope bounds.
Defaults to 0.05.}

\item{nsim}{The number of simulations to perform for envelope construction.
Defaults to 100.}

\item{responses}{An optional list of values to be used as response variables to refit the model.}

\item{no_warnings}{If TRUE, ignore simulations that threw warnings.}

\item{no_messages}{If TRUE, ignore simulations that shown messages.}

\item{converged_only}{Use p-values from converged models only.}

\item{show_progress}{Display a progress bar for the simulation iteration.}

\item{plot.it}{Logical. Generate envelope plot.}

\item{refit_fn}{Function to refit the model with new responses. If \code{NULL},
defaults to \code{get_refit(model, y, ...)}.}

\item{...}{Extra arguments to \code{\link[=get_refit]{get_refit()}}}
}
\value{
An object of class \code{AD_envelope}, which contains the following components:
\describe{
\item{expected}{A vector of expected quantiles from a normal distribution.}
\item{observed}{A vector of observed quantiles from the model residuals.}
\item{outside}{A logical vector indicating whether each observation falls
outside the constructed envelope bounds.}
\item{lower}{The lower bounds of the envelope for each observation.}
\item{med}{The median bounds of the envelope for each observation.}
\item{upper}{The upper bounds of the envelope for each observation.}
}
}
\description{
Generates QQ-plot with simulated envelope residuals.
}
\details{
Simulates new responses using \code{\link[stats:simulate]{stats::simulate()}} and refits the model
for each vector of new responses using \code{\link[=get_refit]{get_refit()}}. The function then computes
residuals for each simulation, sorts them, and constructs envelope bands and
a median line based on the quantiles of these residuals.

\code{refit_fn} is a function that supposedly compute the refit of \code{model}.
Use this method if the default \code{\link[=get_refit]{get_refit()}} doesn't work.
If \code{refit_fn} is \code{NULL}, it's value is defined as \code{function(y, ...) get_refit(model, y, ...)}.
}
\examples{
fit <- lm(mpg ~ cyl, data = mtcars)

envelope(fit)

# Use pearson residuals, and plot it agains the expected normal quantiles.
env_measures <- envelope(fit,
  residual_fn = function(x) residuals.lm(x, type = "pearson"), plot.it = FALSE
)
plot(env_measures, distribution = stats::qnorm, colors = c("gray", "black"))

## Using custom refit_fn
if (require("survival")) {
  fit <- survreg(Surv(futime, fustat) ~ ecog.ps + rx, ovarian,
    dist = "exponential"
  )
  fitted_rate <- 1 / fitted(fit)
  new_responses <- replicate(100, rexp(length(fitted_rate), fitted_rate), simplify = FALSE)
  refit_surv_ovarian <- function(.y) {
    survreg(Surv(.y, fustat) ~ ecog.ps + rx, ovarian, dist = "exponential")
  }
  env_measures <- envelope(fit,
    responses = new_responses,
    residual_fn = function(x) abs(residuals(x, type = "deviance")),
    refit_fn = refit_surv_ovarian, plot.it = FALSE
  )
  # Absolute residuals are best shown with halfnormal quantiles
  plot(env_measures, distribution = function(p) qnorm((1 + p) / 2))
}

}
\seealso{
\code{\link{get_refit}}, \code{\link{simulate}}, \code{\link{rstudent}}, \code{\link{plot.AD_envelope}},
\code{\link[=parametric_bootstrap]{parametric_bootstrap()}}
}
