% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/particle_filter.R
\name{particle_filter}
\alias{particle_filter}
\title{Particle Filter}
\usage{
particle_filter(
  y,
  num_particles,
  init_fn,
  transition_fn,
  log_likelihood_fn,
  obs_times = NULL,
  algorithm = c("SISAR", "SISR", "SIS"),
  resample_fn = c("stratified", "systematic", "multinomial"),
  threshold = NULL,
  return_particles = TRUE,
  ...
)
}
\arguments{
\item{y}{A numeric vector or matrix of observations. Each row represents an
observation at a time step.}

\item{num_particles}{A positive integer specifying the number of particles.}

\item{init_fn}{A function that initializes the particle states. It should
take the current particles as its first argument and return
a vector or matrix of initial particle states.}

\item{transition_fn}{A function describing the state transition model. It
should take the current particles and the current time step as arguments and
return the propagated particles.}

\item{log_likelihood_fn}{A function that computes the log likelihoods for the
particles. It should accept an observation, the current particles, and the
current time step as arguments and return a numeric vector of log likelihood
values.}

\item{obs_times}{A numeric vector indicating the time points at which
observations in \code{y} are available. Must be of the same length as the
number of rows in \code{y}. If not specified, it is assumed that observations
are available at consecutive time steps, i.e., \code{obs_times = 1:nrow(y)}.}

\item{algorithm}{A character string specifying the particle filtering
algorithm to use. Must be one of \code{"SISAR"}, \code{"SISR"}, or
\code{"SIS"}. Defaults to \code{"SISAR"}.}

\item{resample_fn}{A character string specifying the resampling method.
Must be one of \code{"stratified"}, \code{"systematic"}, or
\code{"multinomial"}. Defaults to \code{"stratified"}.}

\item{threshold}{A numeric value specifying the ESS threshold for triggering
resampling in the \code{"SISAR"} algorithm. If not provided, it defaults to
\code{particles / 2}.}

\item{return_particles}{A logical value indicating whether to return the full
particle history. Defaults to \code{TRUE}.}

\item{...}{Additional arguments passed to \code{init_fn},
\code{transition_fn}, and \code{log_likelihood_fn}.}
}
\value{
A list containing:
  \describe{
    \item{state_est}{A numeric vector of estimated states over
    time, computed as the weighted average of particles.}
    \item{ess}{A numeric vector of the Effective Sample Size (ESS) at each
    time step.}
    \item{loglike}{The accumulated log-likelihood of the observations given
    the model.}
    \item{loglike_history}{A numeric vector of the log-likelihood at each
    time step.}
    \item{algorithm}{A character string indicating the filtering algorithm
    used.}
    \item{particles_history}{(Optional) A list of particle state matrices
    over time (one per time step), returned if \code{return_particles} is
    \code{TRUE}.}
    \item{weights_history}{(Optional) A list of particle weight vectors over
    time (one per time step), returned if \code{return_particles} is
    \code{TRUE}.}
  }
}
\description{
This function implements a bootstrap particle filter for estimating the
hidden states in a state space model using sequential Monte Carlo methods.
Three filtering variants are supported:
\enumerate{
  \item \strong{SIS:} Sequential Importance Sampling (without resampling).
  \item \strong{SISR:} Sequential Importance Sampling with resampling at
  every time step.
  \item \strong{SISAR:} SIS with adaptive resampling based on the Effective
  Sample Size (ESS). Resampling is triggered when the ESS falls below a
  given threshold (default \code{particles / 2}).
}
It is recommended to use either SISR or SISAR to avoid weight degeneracy.
}
\details{
The particle filter is a sequential Monte Carlo method that approximates the
posterior distribution of the state in a state space model. The three
supported algorithms differ in their approach to resampling:
\enumerate{
  \item \strong{SIS:} Particles are propagated and weighted without any
   resampling, which may lead to weight degeneracy over time.
  \item \strong{SISR:} Resampling is performed at every time step to combat
  weight degeneracy.
  \item \strong{SISAR:} Resampling is performed adaptively; particles are
  resampled only when the Effective Sample Size (ESS) falls below a
  specified threshold (defaulting to \code{particles / 2}).
}
The Effective Sample Size (ESS) in context of particle filters is defined as
\deqn{ESS = \left(\sum_{i=1}^{\text{n}} w_i^2\right)^{-1},}
where \eqn{n} is the number of particles and \eqn{w_i} are the
normalized weights of the particles.

The default resampling method is stratified resampling, as Douc et al., 2005
showed that it always gives a lower variance compared to
multinomial resampling.
}
\examples{
init_fn <- function(particles) rnorm(particles, 0, 1)
transition_fn <- function(particles) particles + rnorm(length(particles))
log_likelihood_fn <- function(y, particles) {
  dnorm(y, mean = particles, sd = 1, log = TRUE)
}

# Generate data
y <- cumsum(rnorm(50))
num_particles <- 100

# Run the particle filter using default settings.
result <- particle_filter(
  y = y,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn
)
plot(result$state_est, type = "l", col = "blue", main = "State Estimates")

# With parameters
init_fn <- function(particles) rnorm(particles, 0, 1)
transition_fn <- function(particles, mu) {
  particles + rnorm(length(particles), mean = mu)
}
log_likelihood_fn <- function(y, particles, sigma) {
  dnorm(y, mean = particles, sd = sigma, log = TRUE)
}

# Generate data
y <- cumsum(rnorm(50))
num_particles <- 100

# Run the particle filter using default settings.
result <- particle_filter(
  y = y,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn,
  mu = 1,
  sigma = 1
)
plot(result$state_est, type = "l", col = "blue", main = "State Estimates")

# With observations gaps
init_fn <- function(particles) rnorm(particles, 0, 1)
transition_fn <- function(particles, mu) {
  particles + rnorm(length(particles), mean = mu)
}
log_likelihood_fn <- function(y, particles, sigma) {
  dnorm(y, mean = particles, sd = sigma, log = TRUE)
}

# Generate data using DGP
simulate_ssm <- function(num_steps, mu, sigma) {
  x <- numeric(num_steps)
  y <- numeric(num_steps)
  x[1] <- rnorm(1, mean = 0, sd = sigma)
  y[1] <- rnorm(1, mean = x[1], sd = sigma)
  for (t in 2:num_steps) {
    x[t] <- mu * x[t - 1] + sin(x[t - 1]) + rnorm(1, mean = 0, sd = sigma)
    y[t] <- x[t] + rnorm(1, mean = 0, sd = sigma)
  }
  y
}

data <- simulate_ssm(10, mu = 1, sigma = 1)
# Suppose we have data for t=1,2,3,5,6,7,8,9,10 (i.e., missing at t=4)

obs_times <- c(1, 2, 3, 5, 6, 7, 8, 9, 10)
data <- data[obs_times]

num_particles <- 100
# Run the particle filter
# Specify observation times in the particle filter using obs_times
result <- particle_filter(
  y = data,
  num_particles = num_particles,
  init_fn = init_fn,
  transition_fn = transition_fn,
  log_likelihood_fn = log_likelihood_fn,
  obs_times = obs_times,
  mu = 1,
  sigma = 1,
)
plot(result$state_est, type = "l", col = "blue", main = "State Estimates")
}
\references{
Douc, R., Cappé, O., & Moulines, E. (2005). Comparison of
Resampling Schemes for Particle Filtering.
Accessible at: https://arxiv.org/abs/cs/0507025
}
