% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lm_b.R
\name{lm_b}
\alias{lm_b}
\title{Bayesian Linear Models}
\usage{
lm_b(
  formula,
  data,
  weights,
  prior = c("zellner", "conjugate", "improper")[1],
  zellner_g,
  prior_beta_mean,
  prior_beta_precision,
  prior_var_shape,
  prior_var_rate,
  ROPE,
  CI_level = 0.95
)
}
\arguments{
\item{formula}{A formula specifying the model.}

\item{data}{A data frame in which the variables specified in the formula
will be found. If missing, the variables are searched for in the standard way.
However, it is strongly recommended that you use this argument so that other
generics for bayesics objects work correctly.}

\item{weights}{an optional vector of weights to be used in the fitting process.
Should be NULL or a numeric vector. If non-NULL, it is assumed that the
variance of \eqn{y_i} can be written as \eqn{Var(y_i) = \sigma^2/w_i}. While the
estimands remain the same, the estimation is done by performing unweighted
lm_b on \eqn{W^{\frac{1}{2}}y} and \eqn{W^{\frac{1}{2}}X}, where \eqn{W} is the
diagonal matrix of weights.  Note that this then affects the zellner prior.}

\item{prior}{character.  One of "zellner", "conjugate", or "improper", giving
the type of prior used on the regression coefficients.}

\item{zellner_g}{numeric.  Positive number giving the value of "g" in Zellner's
g prior.  Ignored unless prior = "zellner".}

\item{prior_beta_mean}{numeric vector of same length as regression coefficients
(denoted p). Unless otherwise specified, automatically set to rep(0,p).  Ignored
unless prior = "conjugate".}

\item{prior_beta_precision}{pxp matrix giving a priori precision matrix to be
scaled by the residual precision.}

\item{prior_var_shape}{numeric. Twice the shape parameter for the inverse gamma prior on
the residual variance(s).  I.e., \eqn{\sigma^2\sim IG}(prior_var_shape/2,prior_var_rate/2).}

\item{prior_var_rate}{numeric. Twice the rate parameter for the inverse gamma prior on
the residual variance(s).  I.e., \eqn{\sigma^2\sim IG}(prior_var_shape/2,prior_var_rate/2).}

\item{ROPE}{vector of positive values giving ROPE boundaries for each regression
coefficient.  Optionally, you can not include a ROPE boundary for the intercept.
If missing, defaults go to those suggested by Kruchke (2018).}

\item{CI_level}{numeric. Credible interval level.}
}
\value{
lm_b() returns an object of class "lm_b", which behaves as a list with
the following elements:
\itemize{
\item summary - tibble giving results for regression coefficients
\item posterior_parameters - list giving the posterior parameters
\item hyperparameters - list giving the user input (or default) hyperparameters used
\item fitted - posterior mean of the individuals' means
\item residuals - posterior mean of the residuals
\item formula, data - input by user
}
}
\description{
lm_b is used to fit linear models.  It can be used to carry out
regression, single stratum analysis of variance and analysis of
covariance (although aov_b may provide a more convenient
interface for ANOVA.)
}
\details{
\strong{MODEL:}

The likelihood is given by
\deqn{
 y_i \overset{ind}{\sim} N(x_i'\beta,\sigma^2).
}
The prior is given by
\deqn{
 \beta|\sigma^2 \sim N\left( \mu, \sigma^2 V^{-1} \right) \\
 \sigma^2 \sim \Gamma^{-1}(a/2,b/2).
}
\itemize{
\item For Zellner's g prior, \eqn{\frac{1}{g}X'X}.
\item The default for the conjugate prior is based on arguments from
standardized regression.  The default \eqn{V} is set according to the following: "a priori,
we are 95\% certain that a standard deviation increase in \eqn{X} will not lead
to more than a 5 standard deviation in the mean of \eqn{y}."  If we then
set the prior on the intercept to be flat, this leads to
\deqn{
   V^{1/2} = diag(1,s_{x_1},\ldots,s_{x_p}) / 2.5,
 }
where \eqn{s_y} is the standard deviation of \eqn{y}, and \eqn{s_{x_j}} is
the standard deviation of the \eqn{j^{th}} covariate.
\item Unless \code{prior_var_shape} AND \code{prior_var_rate} are provided,
the inverse gamma prior on the residual variance will place 50\% prior
probability that the correlation between the response and the fitted values
is between 0.1 and 0.9.
\item If \code{prior = "improper"}, then the prior is
\deqn{
   \pi(\beta,\sigma^2) \propto \frac{1}{\sigma^2}.
 }
}

\strong{ROPE:}

If missing, the ROPE bounds will be given under the principle of "half of a
small effect size."  Using Cohen's D of 0.2 as a small effect size, the ROPE is
built under the principle that moving the full range of X (i.e., \eqn{\pm 2 s_x})
will not move the mean of y by more than the overall mean of \eqn{y}
minus \eqn{0.1s_y} to the overall mean of \eqn{y} plus \eqn{0.1s_y}.
The result is a ROPE equal to \eqn{|\beta_j| < 0.05s_y/s_j}.  If the covariate is
binary, then this is simply \eqn{|\beta_j| < 0.2s_y}.
}
\examples{
\donttest{
# Generate some data
set.seed(2025)
N = 500
test_data = 
  data.frame(x1 = rnorm(N),
             x2 = rnorm(N),
             x3 = letters[1:5])
test_data$outcome = 
  rnorm(N,-1 + test_data$x1 + 2 * (test_data$x3 \%in\% c("d","e")) )

# Find good hyperparameters for the residual variance
s2_hyperparameters = 
  find_invgamma_parms(lower_R2 = 0.05,
                      upper_R2 = 0.95,
                      probability = 0.8,
                      response_variance = var(test_data$outcome))
## Check (should equal ~ 0.8)
extraDistr::pinvgamma((1.0 - 0.05) * var(test_data$outcome),
                      0.5 * s2_hyperparameters[1],
                      0.5 * s2_hyperparameters[2]) -
  extraDistr::pinvgamma((1.0 - 0.95) * var(test_data$outcome),
                        0.5 * s2_hyperparameters[1],
                        0.5 * s2_hyperparameters[2])

# Fit the linear model using a conjugate prior
fit1 <-
  lm_b(outcome ~ x1 + x2 + x3,
       data = test_data,
       prior = "conj",
       prior_var_shape = s2_hyperparameters["shape"],
       prior_var_rate = s2_hyperparameters["rate"])
fit1
summary(fit1,
        CI_level = 0.99)
plot(fit1)
coef(fit1)
credint(fit1,
        CI_level = 0.9)
bayes_factors(fit1)
bayes_factors(fit1,
              by = "var")
AIC(fit1)
BIC(fit1)
DIC(fit1)
WAIC(fit1)
vcov(fit1)
preds = predict(fit1)

# Try other priors
fit2 <-
  lm_b(outcome ~ x1 + x2 + x3,
       data = test_data) # Default is prior = "zellner"
fit3 <-
  lm_b(outcome ~ x1 + x2 + x3,
       data = test_data,
       prior = "improper")
}



}
