% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_savagedickey.R
\name{bayesfactor_savagedickey}
\alias{bayesfactor_savagedickey}
\alias{bayesfactor_savagedickey.numeric}
\alias{bayesfactor_savagedickey.stanreg}
\alias{bayesfactor_savagedickey.brmsfit}
\alias{bayesfactor_savagedickey.emmGrid}
\alias{bayesfactor_savagedickey.data.frame}
\title{Savage-Dickey density ratio Bayes Factor (BF)}
\usage{
bayesfactor_savagedickey(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE, ...)

\method{bayesfactor_savagedickey}{numeric}(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE, ...)

\method{bayesfactor_savagedickey}{stanreg}(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE,
  effects = c("fixed", "random", "all"), ...)

\method{bayesfactor_savagedickey}{brmsfit}(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE,
  effects = c("fixed", "random", "all"), ...)

\method{bayesfactor_savagedickey}{emmGrid}(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE, ...)

\method{bayesfactor_savagedickey}{data.frame}(posterior, prior = NULL,
  direction = "two-sided", hypothesis = 0, verbose = TRUE, ...)
}
\arguments{
\item{posterior}{A numerical vector, \code{stanreg} / \code{brmsfit} object, \code{emmGrid} or a data frame - representing a posterior distribution(s) from (see Details).}

\item{prior}{An object representing a prior distribution (see Details).}

\item{direction}{Test type (see details). One of \code{0}, \code{"two-sided"} (default, two tailed),
\code{-1}, \code{"left"} (left tailed) or \code{1}, \code{"right"} (right tailed).}

\item{hypothesis}{Value to be tested against (usually \code{0} in the context of null hypothesis testing).}

\item{verbose}{Toggle off warnings.}

\item{...}{Currently not used.}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}
}
\value{
A data frame containing the Bayes factor representing evidence \emph{against} the (point) null effect model.
}
\description{
This method computes the ratio between the density of a single value (typically the null)
of two distributions. When the compared distributions are the posterior and the prior distributions,
this results is an approximation of a Bayes factor comparing the model against a model in which
the parameter of choice is restricted to the point null.
\cr \cr
See also \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.
}
\details{
This method is used to compute Bayes factors based on prior and posterior distributions.
When \code{posterior} is a model (\code{stanreg}, \code{brmsfit}), posterior and prior samples are
extracted for each parameter, and Savage-Dickey Bayes factors are computed for each parameter.

\strong{NOTE:} For \code{brmsfit} models, the model must have been fitted with \emph{custom (non-default)} priors. See example below.

\subsection{Setting the correct \code{prior}}{
It is important to provide the correct \code{prior} for meaningful results.
\itemize{
  \item When \code{posterior} is a numerical vector, \code{prior} should also be a numerical vector.
  \item When \code{posterior} is an \code{emmGrid} object based on a \code{stanreg} or \code{brmsfit} model, \code{prior} should be \emph{that model object} (see example).
  \item When \code{posterior} is a \code{stanreg} or \code{brmsfit} model, there is no need to specify \code{prior}, as prior samples are drawn internally.
  \item When \code{posterior} is a \code{data.frame}, \code{prior} should also be a \code{data.frame}, with matching column order.
}}
\subsection{One-sided Tests (setting an order restriction)}{
One sided tests (controlled by \code{direction}) are conducted by setting an order restriction on
the prior and posterior distributions (\cite{Morey & Wagenmakers, 2013}).
}
\subsection{Interpreting Bayes Factors}{
A Bayes factor greater than 1 can be interpereted as evidence against the null,
at which one convention is that a Bayes factor greater than 3 can be considered
as "substantial" evidence against the null (and vice versa, a Bayes factor
smaller than 1/3 indicates substantial evidence in favor of the null-hypothesis)
(\cite{Wetzels et al. 2011}).
}
}
\examples{
library(bayestestR)

prior <- distribution_normal(1000, mean = 0, sd = 1)
posterior <- distribution_normal(1000, mean = .5, sd = .3)

bayesfactor_savagedickey(posterior, prior)
\dontrun{
# rstanarm models
# ---------------
library(rstanarm)
stan_model <- stan_lmer(extra ~ group + (1 | ID), data = sleep)
bayesfactor_savagedickey(stan_model)

# emmGrid objects
# ---------------
library(emmeans)
group_diff <- pairs(emmeans(stan_model, ~group))
bayesfactor_savagedickey(group_diff, prior = stan_model)

# brms models
# -----------
library(brms)
my_custom_priors <-
  set_prior("student_t(3, 0, 1)", class = "b") +
  set_prior("student_t(3, 0, 1)", class = "sd", group = "ID")

brms_model <- brm(extra ~ group + (1 | ID),
  data = sleep,
  prior = my_custom_priors
)
bayesfactor_savagedickey(brms_model)
}

}
\references{
\itemize{
\item Wagenmakers, E. J., Lodewyckx, T., Kuriyal, H., and Grasman, R. (2010). Bayesian hypothesis testing for psychologists: A tutorial on the Savage-Dickey method. Cognitive psychology, 60(3), 158-189.
\item Wetzels, R., Matzke, D., Lee, M. D., Rouder, J. N., Iverson, G. J., and Wagenmakers, E.-J. (2011). Statistical Evidence in Experimental Psychology: An Empirical Comparison Using 855 t Tests. Perspectives on Psychological Science, 6(3), 291–298. \doi{10.1177/1745691611406923}
\item Heck, D. W. (2019). A caveat on the Savage–Dickey density ratio: The case of computing Bayes factors for regression parameters. British Journal of Mathematical and Statistical Psychology, 72(2), 316-333.
\item Morey, R. D., & Wagenmakers, E. J. (2014). Simple relation between Bayesian order-restricted and point-null hypothesis tests. Statistics & Probability Letters, 92, 121-124.
}
}
\author{
Mattan S. Ben-Shachar
}
