#' Estimate Squared Partial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of squared partial correlation coefficients
#'   (\eqn{r^{2}_{p}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r^{2}_{p}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r^{2}_{p}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r^{2}_{p}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r^{2}_{p}}.}
#'     \item{fun}{Function used ("PCorMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of parameter estimates
#' # (use a large R, for example, R = 20000 for actual research)
#' mc <- MC(object, R = 100)
#' # Generate confidence intervals for standardized regression slopes
#' rp <- PCorMC(mc)
#' # Methods --------------------------------------------------------
#' print(rp)
#' summary(rp)
#' coef(rp)
#' vcov(rp)
#' confint(rp, level = 0.95)
#' @export
#' @family Beta Monte Carlo Functions
#' @keywords betaMC pcor
PCorMC <- function(object) {
  stopifnot(
    methods::is(
      object,
      "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  sr <- SCorMC(object)
  sr <- sr$thetahatstar
  rsq <- RSqMC(object)
  rsq <- rsq$thetahatstar
  thetahatstar <- mapply(
    sr = sr,
    rsq = rsq,
    SIMPLIFY = FALSE,
    FUN = function(sr,
                   rsq) {
      return(
        .PCorSq(
          srsq = sr^2,
          rsq = rsq[1]
        )
      )
    }
  )
  est <- .PCorSq(
    srsq = .SPCor(
      betastar = object$lm_process$betastar,
      sigmacapx = object$lm_process$sigmacapx
    )^2,
    rsq = object$lm_process$summary_lm$r.squared
  )
  names(est) <- object$lm_process$xnames
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    est = est,
    fun = "PCorMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
