% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sets.R
\name{sets_read}
\alias{sets_read}
\title{Read sets and calculate concentrations and variability.}
\usage{
sets_read(sets, cal_names, cal_values, exclude_cals = list(),
  additional_vars = c("name"), additional_sep = "_", sep = ",",
  dec = ".", path = ".", file_name = "set_#NUM#.csv",
  model_func = fit_linear, plot_func = plot_linear,
  interpolate_func = interpolate_linear, write_data = TRUE,
  use_written_data = FALSE)
}
\arguments{
\item{sets}{The number of sets (e.g. \code{3`` attempts to read}set_1.csv\code{,}set_2.csv\code{,}set_3.csv\code{), see}file_name`.}

\item{cal_names}{A vector of strings containing the names of the samples used
as calibrators.}

\item{cal_values}{A numeric vector with the known concentrations of those
samples (must be in the same order).}

\item{exclude_cals}{A list of calibrators to exclude, e.g.:
\code{list(set1 = c("CAL1"))}.}

\item{additional_vars}{Vector of strings containing the names for the
additional columns.}

\item{additional_sep}{String / RegExp that separates additional vars, e.g.:
\code{"ID_blue_cold"} with \code{additional_sep = "_"} will be separated
into three columns containing \code{"ID"}, \code{"blue"} and \code{"cold"}.
If the separated data would exceed the columns in \code{additional_vars}
the last column will contain a string with separator (e.g.: \code{"blue_cold"}).
If data is missing \code{NA} is inserted.}

\item{sep}{Separator used in the csv-file, either "," or ";" (see
\code{\link[utils:read.csv]{utils::read.csv()}}).}

\item{dec}{The character used for decimal points (see \code{\link[utils:read.csv]{utils::read.csv()}}).
"AUTO" will result in "." if \code{sep} is "," and "," for ";".}

\item{path}{The path to the file (no trailing "/" or "\" !).}

\item{file_name}{Name of the file from which to read the data. May contain
"#NUM#" as a placeholder if you have multiple files.}

\item{model_func}{A function generating a model to fit the calibrators,
e.g. \code{\link[=fit_linear]{fit_linear()}}, \code{\link[=fit_lnln]{fit_lnln()}}.}

\item{plot_func}{Function used to display the fitted line.}

\item{interpolate_func}{A function used to interpolate the concentrations of
the other samples, based on the model, e.g.
\code{\link[=interpolate_linear]{interpolate_linear()}}, \code{\link[=interpolate_lnln]{interpolate_lnln()}}.}

\item{write_data}{Write the calculated data into \code{data_all.csv} and
\code{data_samples.csv}?}

\item{use_written_data}{Try to read \code{data_all.csv} and \code{data_read.csv}
instead of raw data. Useful if you have to re-run the script, but the raw
data does not change.}
}
\value{
A list:
\itemize{
\item \code{$all}: here you will find all the data , including calibrators,
duplicates, ... (saved in \code{data_all.csv} if \code{write_data = TRUE})
\item \code{$samples}: only one row per distinct sample here - no calibrators, no
duplicates -> most often you will work with this data
(saved in \code{data_samples.csv} if \code{write_data = TRUE})
\item \code{$set1}: a list
\itemize{
\item \code{$plot}: a plot showing you the function used to calculate the
concentrations for this set. The points represent the calibrators.
\item \code{$model}: the model as returned by \code{model_func}
}
\item (\code{$set2} - \code{$setN}): the same information for every set you have
}
}
\description{
Basicaly a wrapper around \code{\link[=set_read]{set_read()}}, \code{\link[=set_calc_concentrations]{set_calc_concentrations()}} and
\code{\link[=set_calc_variability]{set_calc_variability()}}.

For a gentler introduction see examples and Vignette "Introduction".

May write the processed data into two files: \code{data_samples.csv},
\code{data_all.csv}.
}
\examples{
# files "set_1.csv" and "set_2.csv" containing raw values and the
# corresponding lables (consisting of ID and point in time like
# "ID_TIME")
read.csv(
  file = system.file("extdata", "set_1.csv", package = "bioset"),
  header = FALSE,
  colClasses = "character"
)
read.csv(
  file = system.file("extdata", "set_2.csv", package = "bioset"),
  header = FALSE,
  colClasses = "character"
)

# the known concentration of the calibrators contained in these plates
cals <- c(10, 20, 30, 40)      # ng / ml
names(cals) <- c("CAL1", "CAL2", "CAL3", "CAL4")

# read both files into a tibble
# columns "ID" and "time" separated by "_"
# and calculate concentrations using the calibrators
result <- sets_read(
  sets = 2,                      # expect 2 plates
  path = system.file("extdata", package = "bioset"),
  additional_vars = c("ID", "time"),  # expect the labels to contain ID and
                                      # point in time
  additional_sep = "_",               # separated by "_"
  cal_names = names(cals),       # that's what they're called in the files
  cal_values = cals,             # the concentration has to be known
  write_data = FALSE             # do not store the results in csv-files
)

# inspect results (all values contained in the two original files)
result$all
# (all values except CAL1-4)
result$samples
# inspect goodness of fit
# for plate 1
result$set_1$plot
result$set_1$model
# for plate 2
result$set_2$plot
result$set_2$model
}
\seealso{
Other set functions: \code{\link{set_calc_concentrations}},
  \code{\link{set_calc_variability}},
  \code{\link{set_read}}
}
