% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bizicount-methods.R
\name{extract.bizicount}
\alias{extract.bizicount}
\title{Texreg for bizicount objects}
\usage{
\method{extract}{bizicount}(model, CI = NULL, id = TRUE)
}
\arguments{
\item{model}{A \code{\link{bizicount-class}} model object (S3).}

\item{CI}{The two-tailed confidence level, if confidence intervals are
desired in the texreg object, otherwise \code{NULL}.}

\item{id}{Logical indicating whether to prepend equation identifiers to
coefficient names (\code{ct_} for count parameters, \code{zi_} for zero-inflated parameters)}
}
\value{
A \code{\link[texreg]{texreg-class}} object, as produced by
\code{\link[texreg]{createTexreg}}, which can interface with all of that
package's generics.
}
\description{
This is a method for the \code{\link[texreg]{extract}} generic
to be used with objects that are output from the \code{\link{bizicount}}
function. The results can be used with any of the
\code{\link[texreg]{texreg-package}} generics.
}
\note{
Users can typically just call \code{\link[texreg]{texreg}} directly on
a \code{\link{bizicount-class}} object, instead of first extracting and
then calling texreg.
}
\examples{
## SETUP
set.seed(123)
n = 500

# define a function to simulate from a gaussian copula
# first margin is zero-inflated negative binomial (zinb)
# second margin is zero-inflated poisson (zip)
# Note: marginal distributions are hard-coded in function, including
# inverse dispersion parameter for zinb.
gen = function(n,
               b1,
               b2,
               g1,
               g2,
               dep) {

     k1 = length(b1)
     k2 = length(b2)

     X1 = cbind(1, matrix(rbinom(n * (k1 - 1), 1, .5), ncol = k1 - 1))
     X2 = cbind(1, matrix(rexp(n * (k2 - 1), 3), ncol = k2 - 1))

     lam1 = exp(X1 \%*\% b1)
     lam2 = exp(X2 \%*\% b2)

     Z1 = cbind(1, matrix(runif(n * (k1 - 1), -1, 1), ncol = k1 - 1))
     Z2 = cbind(1, matrix(rnorm(n * (k2 - 1)), ncol = k2 - 1))

     psi1 = plogis(Z1 \%*\% g1)
     psi2 = plogis(Z2 \%*\% g2)

     norm_vars = MASS::mvrnorm(
          n,
          mu = c(0, 0),
          Sigma = matrix(c(1, dep, dep, 1), ncol =2)
     )

     U = pnorm(norm_vars)

     y1 =  qzinb(U[, 1],
                 mu = lam1,
                 psi = psi1,
                 size = .3)
     y2 =  qzip(U[, 2],
                lambda = lam2,
                psi = psi2)

     dat = data.frame(
          X1 = X1[, -1],
          X2 = X2[, -1],
          Z1 = Z1[, -1],
          Z2 = Z2[, -1],
          y1,
          y2,
          lam1,
          lam2,
          psi1,
          psi2
     )
     return(dat)
}


# define parameters
b1 = c(1, -2, 3)
b2 = c(-1, 3, 1)
g1 = c(2, -1.5, 2)
g2 = c(-1, -3.75, 1.25)
rho = .5


# generate data
dat = gen(n, b1, b2, g1, g2, rho)
f1 = y1 ~ X1.1 + X1.2 | Z1.1 + Z1.2
f2 = y2 ~ X2.1 + X2.2 | Z2.1 + Z2.2

## END SETUP

# estimate model

mod = bizicount(f1, f2, dat, cop = "g", margins = c("zinb", "zip"), keep=TRUE)

# extract texreg objects, one with SEs, one with CIs
tr_obj_se = texreg::extract(mod)
tr_obj_ci = texreg::extract(mod, CI = .95)


# output to latex, single table.
# Note use of c(), because tr_obj_se, tr_obj_ci are lists.
texreg::texreg(c(tr_obj_se, tr_obj_ci))


# output as plaintext, two tables
texreg::screenreg(tr_obj_se)
texreg::screenreg(tr_obj_ci)
}
\references{
Leifeld, Philip (2013). texreg: Conversion of Statistical Model
Output in R to LaTeX and HTML Tables. Journal of Statistical Software,
55(8), 1-24. URL http://dx.doi.org/10.18637/jss.v055.i08.
}
\seealso{
\code{\link[texreg]{extract}}, \code{\link[texreg]{createTexreg}},
\code{\link[bizicount]{bizicount}}
}
\author{
John Niehaus
}
