#' @title Efficiencies of pairwise differences.
#' 
#' @description
#' Tabulates the efficieny of pairwise treatment difference for each stratum of a design constructed by \code{\link[blocksdesign]{blocks}}.
#' 
#' @details
#'  Tabulates the efficiency factors of the pairwise treatment differences in every statum of any design built by the \code{blocks} function. 
#'  For equi-replicate block designs, the harmonic mean of the pairwise efficiency factors will equal the A-efficiency factor.
#' 
#' @param Design a design data frame generated by \code{\link[blocksdesign]{blocks}}.
#' 
#' @return  
#' \item{Efficiencies}{List of treatments efficiency matrices for all pairwise differences for each stratum of the design}
#' 
#' @examples 
#' 
#' # 4 replicates of 50 treatments in complete randomized blocks 
#' efficiencies(blocks(treatments=50,replicates=4,blocklevels=c(4,5))$Design)
#' 
#' @export
efficiencies=function(Design){
  nunits=nrow(Design)	
  strata=ncol(Design)-2
  pairwise_effics=vector(mode="list",length=strata)	
  TF=as.factor(Design$Treatments)
  T=matrix(0,nrow=nunits,ncol=nlevels(TF))	
  T[cbind(rep(1:nunits),TF)]=1
  r=apply(T, 2, sum)		
  for (i in 1:strata) {
    BF=as.factor(Design[,i])
    B=matrix(0,nrow=nunits,ncol=nlevels(BF))
    B[cbind(rep(1:nunits),BF)]=1
    B=B[ , (1:(ncol(B)-1)),drop=FALSE ]
    B=crossprod(t(B),diag(1/sqrt(apply(B, 2, sum)),nrow=ncol(B)))
    V=solve(diag(r,nrow = length(r))-crossprod(crossprod(B,T)))
    D= crossprod( t(rep(1,nlevels(TF)))  ,t(diag(V)))   +  crossprod( t(diag(V)), t(rep(1,nlevels(TF))) ) - 2*V		
    N= crossprod( t(rep(1,nlevels(TF)))  ,1/t(r))       +  crossprod( 1/t(r), t(rep(1,nlevels(TF))))
    E=N/D
    E[upper.tri(E,diag=TRUE)] = NA
    pairwise_effics[[i]]=E
  }
  pairwise_effics
} 
