% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/02-pipeline.R
\name{pipeline}
\alias{pipeline}
\alias{br_pipeline}
\alias{br_set_y}
\alias{br_set_x}
\alias{br_set_x2}
\alias{br_set_model}
\alias{br_run}
\title{Modeling and analysis pipeline}
\usage{
br_pipeline(
  data,
  y,
  x,
  method,
  x2 = NULL,
  group_by = NULL,
  n_workers = 1L,
  run_parallel = lifecycle::deprecated(),
  dry_run = FALSE,
  model_args = list(),
  run_args = list(),
  filter_x = FALSE,
  filter_na_prop = 0.8,
  filter_sd_min = 1e-06,
  filter_var_min = 1e-06,
  filter_min_levels = 2
)

br_set_y(obj, y)

br_set_x(
  obj,
  ...,
  filter_x = FALSE,
  filter_na_prop = 0.8,
  filter_sd_min = 1e-06,
  filter_var_min = 1e-06,
  filter_min_levels = 2
)

br_set_x2(obj, ...)

br_set_model(obj, method, ...)

br_run(
  obj,
  ...,
  group_by = NULL,
  n_workers = 1L,
  run_parallel = lifecycle::deprecated()
)
}
\arguments{
\item{data}{A \code{data.frame} containing all necessary variables for analysis.
Column names should follow R's naming conventions.}

\item{y}{Character vector specifying dependent variables (response variables).
For GLM models, this is typically a single character (e.g., \code{"outcome"}).
For Cox-PH models, it should be a length-2 vector in the format \code{c("time", "status")}.}

\item{x}{Character vector specifying focal independent terms (predictors).}

\item{method}{Method for model construction.
A name or a list specifying custom model setting.
A string representing a complex method setting is acceptable,
e.g., 'quasi(variance = "mu", link = "log")'.
Or a list with 4 elements, see \code{\link[=br_avail_method_config]{br_avail_method_config()}}
for examples.}

\item{x2}{Character vector specifying control independent terms (predictors, optional).}

\item{group_by}{A string specifying the group by column.}

\item{n_workers, run_parallel}{Integer, indicating integer number
of workers to launch, default is \code{1L}. When \verb{>1},
run modeling code in parallel in the background.}

\item{dry_run}{If \code{TRUE}, generates modeling descriptions without
executing the run. Use this to inspect the construction first.}

\item{model_args}{A list of arguments passed to \code{br_set_model()}.}

\item{run_args}{A list of arguments passed to \code{br_run()}.}

\item{filter_x}{Logical, whether to enable pre-filtering of focal variables. Default is \code{FALSE}.}

\item{filter_na_prop}{Numeric, maximum proportion of NA values allowed for a variable. Default is \code{0.8}.}

\item{filter_sd_min}{Numeric, minimum standard deviation required for a variable. Default is \code{1e-6}.}

\item{filter_var_min}{Numeric, minimum variance required for a variable. Default is \code{1e-6}.}

\item{filter_min_levels}{Numeric, minimum number of unique levels required for categorical variables. Default is \code{2}.}

\item{obj}{An object of class \code{breg}.}

\item{...}{Additional arguments depending on the called function.
\itemize{
\item \code{br_set_x()} for passing focal terms as characters.
\item \code{br_set_x2()} for passing control terms as characters.
\item \code{br_set_model()} for passing other configurations for modeling.
\item \code{br_run()} for passing other configurations for obtaining modeling results with \code{\link[broom.helpers:tidy_plus_plus]{broom.helpers::tidy_plus_plus()}}.
e.g., The default value for \code{exponentiate} is \code{FALSE} (coefficients are not exponentiated).
For logistic, and Cox-PH regressions models, \code{exponentiate} is set to \code{TRUE} at default.
}}
}
\value{
An object of class \code{breg} with input values added to corresponding slot(s).
For \code{br_run()}, the returned object is a \code{breg} object with results added to
the slots \verb{@results} and \verb{@results_tidy}, note that \verb{@models} is updated to a list
of constructed model object (See \link{accessors}).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Provides a set of functions for running batch regression analysis.
Combines data setup, model configuration, and execution steps into a single workflow.
Supports both GLM and Cox-PH models with options for focal/control terms and parallel processing.
}
\details{
Please note the difference between \href{https://easystats.github.io/insight/#variables}{variables} and
\href{https://easystats.github.io/insight/#terms}{terms},
e.g., \code{x + poly(x, 2)} has \emph{one} variable \code{x}, but \emph{two} terms \code{x} and \code{poly(x, 2)}.
\subsection{Global options}{

\strong{bregr} supported global options can be set with \code{options()}.
Currently they are used in \code{br_run()}.
\itemize{
\item \code{bregr.save_model}: If \code{TRUE}, saves models to local disk.
\item \code{bregr.path}: A path for saving models, defaults to using a
temporary directory.
}
}
}
\section{Functions}{
\itemize{
\item \code{br_pipeline()}: All-in-one end to end wrapper to run the regression analysis in batch.
Which could be splitted into the following steps

\item \code{br_set_y()}: Set dependent variables for model construction.

\item \code{br_set_x()}: Set focal terms for model construction.

\item \code{br_set_x2()}: Set control terms for model construction (Optional in pipeline).

\item \code{br_set_model()}: Set model configurations.

\item \code{br_run()}: Run the regression analysis in batch.

}}
\examples{
library(bregr)
# 1. Pipeline -------------------------
# 1.1. A single linear model ----------
m <- breg(mtcars) |> # set model data
  br_set_y("mpg") |> # set dependent variable
  br_set_x("qsec") |> # set focal variables
  br_set_model("gaussian") |> # set model
  br_run() # run analysis

# get model tidy result
br_get_results(m, tidy = TRUE)
# or m@results_tidy

# compare with R's built-in function
lm(mpg ~ qsec, data = mtcars) |> summary()
# 1.2. Batch linear model -------------
# control variables are injected in all constructed models
# focal variables are injected in constructed models one by one
m2 <- breg(mtcars) |>
  br_set_y("mpg") |>
  br_set_x(colnames(mtcars)[2:4]) |> # set focal variables
  br_set_x2("vs") |> # set control variables
  br_set_model("gaussian") |>
  br_run()
# 1.3. Group by model -------------
m3 <- breg(mtcars) |>
  br_set_y("mpg") |>
  br_set_x("cyl") |>
  br_set_x2("wt") |> # set control variables
  br_set_model("gaussian") |>
  br_run(group_by = "am")

# 2. All-in-one pipeline wrapper ---
m4 <- br_pipeline(mtcars,
  y = "mpg",
  x = colnames(mtcars)[2:4],
  x2 = "vs",
  method = "gaussian"
)

# 3. Customized model -----------
\dontrun{
dt <- data.frame(x = rnorm(100))
dt$y <- rpois(100, exp(1 + dt$x))

m5 <- breg(dt) |>
  br_set_y("y") |>
  br_set_x("x") |>
  br_set_model(method = 'quasi(variance = "mu", link = "log")') |>
  br_run()
}

}
\seealso{
\link{accessors} for accessing \code{breg} object properties.
}
