#' @title Fit an MMRM.
#' @export
#' @family models
#' @description Fit an MMRM model using `brms`.
#' @inheritSection brm_formula Parameterization
#' @return A fitted model object from `brms`.
#' @inheritParams brm_formula
#' @param formula An object of class `"brmsformula"` from [brm_formula()]
#'   or `brms::brmsformula()`. Should include the full mapping
#'   of the model, including fixed effects, residual correlation,
#'   and heterogeneity in the discrete-time-specific residual variance
#'   components.
#' @param ... Arguments to `brms::brm()` other than `data`, `formula`,
#'   and `prior`.
#' @param prior Either `NULL` for default priors
#'   or a `"brmsprior"` object from `brms::prior()`.
#' @param family A `brms` family object generated by [brms::brmsfamily()].
#'   Must fit a continuous outcome variable and have the identity link.
#' @examples
#' if (identical(Sys.getenv("BRM_EXAMPLES", unset = ""), "true")) {
#' set.seed(0L)
#' data <- brm_data(
#'   data = brm_simulate_simple()$data,
#'   outcome = "response",
#'   role = "response",
#'   group = "group",
#'   time = "time",
#'   patient = "patient",
#'   reference_group = "group_1",
#'   reference_time = "time_1"
#' )
#' formula <- brm_formula(
#'   data = data,
#'   baseline = FALSE,
#'   baseline_time = FALSE
#' )
#' # Optional: set the contrast option, which determines the model matrix.
#' options(contrasts = c(unordered = "contr.SAS", ordered = "contr.poly"))
#' # See the fixed effect mapping you get from the data:
#' head(brms::make_standata(formula = formula, data = data)$X)
#' # Specify a different contrast method to use an alternative
#' # mapping when fitting the model with brm_model():
#' options(
#'   contrasts = c(unordered = "contr.treatment", ordered = "contr.poly")
#' )
#' # different model matrix than before:
#' head(brms::make_standata(formula = formula, data = data)$X)
#' tmp <- utils::capture.output(
#'   suppressMessages(
#'     suppressWarnings(
#'       model <- brm_model(
#'         data = data,
#'         formula = formula,
#'         chains = 1,
#'         iter = 100,
#'         refresh = 0
#'       )
#'     )
#'   )
#' )
#' # The output model is a brms model fit object.
#' suppressWarnings(print(model))
#' # The `prior_summary()` function shows the full prior specification
#' # which reflects the fully realized fixed effects mapping.
#' brms::prior_summary(model)
#' }
brm_model <- function(
  data,
  formula,
  ...,
  prior = NULL,
  family = brms::brmsfamily(family = "gaussian")
) {
  brm_data_validate(data)
  brm_formula_validate(formula)
  brms_model_validate_family(family)
  assert(
    inherits(prior %|||% brms::prior("normal(0, 1)"), "brmsprior"),
    message = "prior arg must be a \"brmsprior\" object or NULL."
  )
  model <- brms::brm(
    data = data[!is.na(data[[attr(data, "brm_outcome")]]), ],
    formula = formula,
    prior = prior,
    ...
  )
  model <- brm_model_new(model, formula)
  brm_model_validate(model)
  model
}

brm_model_new <- function(model, formula) {
  structure(
    model,
    class = unique(c("brms_mmrm_model", class(model)))
  )
}

brm_model_validate <- function(model) {
  assert(
    model,
    inherits(., "brms_mmrm_model"),
    inherits(., "brmsfit"),
    message = paste(
      "Please use brms.mmrm::brm_model() to fit the model.",
      "Otherwise, functions like brm_marginal_draws()",
      "in brms.mmrm may not be statistically valid."
    )
  )
}

brms_model_validate_family <- function(family) {
  assert(
    inherits(family, "brmsfamily"),
    message = "family must come from brms::brmsfamily()"
  )
  assert(
    identical(family$type, "real"),
    message = "family must be for continuous outcomes"
  )
  assert(
    identical(family$family, "gaussian"),
    message = "family must be gaussian"
  )
  assert(
    identical(family$link, "identity"),
    message = "family must have the identity link"
  )
  assert(
    identical(family$link_sigma, "log"),
    message = "family must link_sigma equal to 'log'"
  )
}
