% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brm_prior_label.R
\name{brm_prior_label}
\alias{brm_prior_label}
\title{Label a prior with levels in the data.}
\usage{
brm_prior_label(label = NULL, code, group, subgroup = NULL, time)
}
\arguments{
\item{label}{A \code{tibble} with the prior labeling scheme so far,
with one row per model parameter and columns for the
Stan code, treatment group, subgroup, and discrete time point
of each parameter.}

\item{code}{Character of length 1, Stan code for the prior. Could be
a string like \code{"normal(1, 2.2)"}. The full set of priors is given
in the Stan Reference Manual, available from \url{https://mc-stan.org/}.
See the documentation \code{\link[brms:set_prior]{brms::set_prior()}} for more details.}

\item{group}{Value of length 1, level of the treatment group column
in the data to label the prior. The treatment group column
is the one you identified with the \code{group} argument of \code{\link[=brm_data]{brm_data()}}.}

\item{subgroup}{Value of length 1, level of the subgroup column
in the data to label the prior. The subgroup column
is the one you identified with the \code{subgroup} argument of \code{\link[=brm_data]{brm_data()}},
if applicable. Not every dataset has a subgroup variable.
If yours does not, please either ignore this argument or set it to
\code{NULL}.}

\item{time}{Value of length 1, level of the discrete time column
in the data to label the prior. The discrete time column
is the one you identified with the \code{time} argument of \code{\link[=brm_data]{brm_data()}}.}
}
\value{
A \code{tibble} with one row per model parameter and columns for the
Stan code, treatment group, subgroup, and discrete time point
of each parameter. You can supply this \code{tibble} to the \code{label}
argument of \code{\link[=brm_prior_archetype]{brm_prior_archetype()}}.
}
\description{
Label an informative prior for a parameter
using a collection of levels in the data.
}
\section{Prior labeling}{

Informative prior archetypes use a labeling scheme to assign priors
to fixed effects. How it works:

\if{html}{\out{<div class="sourceCode">}}\preformatted{1. First, assign the prior of each parameter a collection
  of labels from the data. This can be done manually or with
  successive calls to [brm_prior_label()].
2. Supply the labeling scheme to [brm_prior_archetype()].
  [brm_prior_archetype()] uses attributes of the archetype
  to map labeled priors to their rightful parameters in the model.
}\if{html}{\out{</div>}}

For informative prior archetypes, this process is much more convenient
and robust than manually calling \code{\link[brms:set_prior]{brms::set_prior()}}.
However, it requires an understanding of how the labels of the priors
map to parameters in the model. This mapping varies from archetype
to archetype, and it is documented in the help pages of
archetype-specific functions such as \code{\link[=brm_archetype_successive_cells]{brm_archetype_successive_cells()}}.
}

\examples{
set.seed(0L)
data <- brm_simulate_outline(
  n_group = 2,
  n_patient = 100,
  n_time = 3,
  rate_dropout = 0,
  rate_lapse = 0
) |>
  dplyr::mutate(response = rnorm(n = dplyr::n())) |>
  brm_simulate_continuous(names = c("biomarker1", "biomarker2")) |>
  brm_simulate_categorical(
    names = c("status1", "status2"),
    levels = c("present", "absent")
  )
archetype <- brm_archetype_successive_cells(data)
dplyr::distinct(data, group, time)
label <- NULL |>
  brm_prior_label("normal(1, 1)", group = "group_1", time = "time_1") |>
  brm_prior_label("normal(1, 2)", group = "group_1", time = "time_2") |>
  brm_prior_label("normal(1, 3)", group = "group_1", time = "time_3") |>
  brm_prior_label("normal(2, 1)", group = "group_2", time = "time_1") |>
  brm_prior_label("normal(2, 2)", group = "group_2", time = "time_2") |>
  brm_prior_label("normal(2, 3)", group = "group_2", time = "time_3")
label
}
\seealso{
Other priors: 
\code{\link{brm_prior_archetype}()},
\code{\link{brm_prior_simple}()}
}
\concept{priors}
