% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bruceR-stats_1_basic.R
\name{TTEST}
\alias{TTEST}
\title{One-sample, independent-samples, and paired-samples t-test.}
\usage{
TTEST(
  data,
  y,
  x = NULL,
  paired = FALSE,
  var.equal = TRUE,
  mean.diff = TRUE,
  test.value = 0,
  test.sided = c("=", "<", ">"),
  factor.rev = TRUE,
  bayes.prior = "medium",
  digits = 2,
  nsmall = digits,
  file = NULL
)
}
\arguments{
\item{data}{Data frame (wide-format only, i.e., one case in one row).}

\item{y}{Dependent variable(s).
Multiple variables should be included in a character vector \code{c()}.

For paired-samples \emph{t}-test, the number of variables should be 2, 4, 6, etc.}

\item{x}{Independent variable(s).
Multiple variables should be included in a character vector \code{c()}.

Only necessary for independent-samples \emph{t}-test.}

\item{paired}{For paired-samples \emph{t}-test, set it to \code{TRUE}. Default is \code{FALSE}.}

\item{var.equal}{If Levene's test indicates a violation of the homogeneity of variance,
then you should better set this argument to \code{FALSE}. Default is \code{TRUE}.}

\item{mean.diff}{Whether to display results of mean difference and its 95\% CI. Default is \code{TRUE}.}

\item{test.value}{The true value of the mean (or difference in means for a two-samples test). Default is \code{0}.}

\item{test.sided}{Any of \code{"="} (two-sided, the default), \code{"<"} (one-sided), or \code{">"} (one-sided).}

\item{factor.rev}{Whether to reverse the levels of factor (X)
such that the test compares higher vs. lower level. Default is \code{TRUE}.}

\item{bayes.prior}{Prior scale in Bayesian \emph{t}-test. Default is 0.707.
See details in \code{\link[BayesFactor:ttestBF]{BayesFactor::ttestBF()}}.}

\item{digits, nsmall}{Number of decimal places of output. Default is \code{2}.}

\item{file}{File name of MS Word (\code{.doc}).}
}
\description{
One-sample, independent-samples, and paired-samples \emph{t}-test,
with both Frequentist and Bayesian approaches.
The output includes descriptives, \emph{t} statistics,
mean difference with 95\% CI, Cohen's \emph{d} with 95\% CI, and Bayes factor (BF10).
It also tests the assumption of homogeneity of variance
and allows users to determine whether variances are equal or not.

Users can simultaneously test multiple dependent and/or independent variables.
The results of one pair of Y-X would be summarized in one row in the output.
Key results can be saved in APA format to MS Word.
}
\details{
Note that the point estimate of Cohen's \emph{d} is computed using
the common method "Cohen's \emph{d} = mean difference / (pooled) standard deviation", which is
consistent with results from other R packages (e.g., \code{effectsize}) and software (e.g., \code{jamovi}).
The 95\% CI of Cohen's \emph{d} is estimated based on the 95\% CI of mean difference
(i.e., also divided by the pooled standard deviation).

However, different packages and software diverge greatly on the estimate of the 95\% CI of Cohen's \emph{d}.
R packages such as \code{psych} and \code{effectsize}, R software \code{jamovi},
and several online statistical tools for estimating effect sizes
indeed produce surprisingly inconsistent results on the 95\% CI of Cohen's \emph{d}.

See an illustration of this issue in the section "Examples".
}
\examples{
## Demo data ##
d1 = between.3
d1$Y1 = d1$SCORE  # shorter name for convenience
d1$Y2 = rnorm(32)  # random variable
d1$B = factor(d1$B, levels=1:2, labels=c("Low", "High"))
d1$C = factor(d1$C, levels=1:2, labels=c("M", "F"))
d2 = within.1

## One-sample t-test ##
TTEST(d1, "SCORE")
TTEST(d1, "SCORE", test.value=5)

## Independent-samples t-test ##
TTEST(d1, "SCORE", x="A")
TTEST(d1, "SCORE", x="A", var.equal=FALSE)
TTEST(d1, y="Y1", x=c("A", "B", "C"))
TTEST(d1, y=c("Y1", "Y2"), x=c("A", "B", "C"),
      mean.diff=FALSE,  # remove to save space
      file="t-result.doc")
unlink("t-result.doc")  # delete file for code check

## Paired-samples t-test ##
TTEST(d2, y=c("A1", "A2"), paired=TRUE)
TTEST(d2, y=c("A1", "A2", "A3", "A4"), paired=TRUE)


\dontrun{

  ## Illustration for the issue stated in "Details"

  # Inconsistency in the 95\% CI of Cohen's d between R packages:
  # In this example, the true point estimate of Cohen's d = 3.00
  # and its 95\% CI should be equal to 95\% CI of mean difference.

  data = data.frame(X=rep(1:2, each=3), Y=1:6)
  data  # simple demo data

  TTEST(data, y="Y", x="X")
  # d = 3.00 [0.73, 5.27] (estimated based on 95\% CI of mean difference)

  MANOVA(data, dv="Y", between="X") \%>\%
    EMMEANS("X")
  # d = 3.00 [0.73, 5.27] (the same as TTEST)

  psych::cohen.d(x=data, group="X")
  # d = 3.67 [0.04, 7.35] (strange)

  psych::d.ci(d=3.00, n1=3, n2=3)
  # d = 3.00 [-0.15, 6.12] (significance inconsistent with t-test)

  # jamovi uses psych::d.ci() to compute 95\% CI
  # so its results are also: 3.00 [-0.15, 6.12]

  effectsize::cohens_d(Y ~ rev(X), data=data)
  # d = 3.00 [0.38, 5.50] (using the noncentrality parameter method)

  effectsize::t_to_d(t=t.test(Y ~ rev(X), data=data, var.equal=TRUE)$statistic,
                     df_error=4)
  # d = 3.67 [0.47, 6.74] (merely an approximate estimate, often overestimated)
  # see ?effectsize::t_to_d

  # https://www.psychometrica.de/effect_size.html
  # d = 3.00 [0.67, 5.33] (slightly different from TTEST)

  # https://www.campbellcollaboration.org/escalc/
  # d = 3.00 [0.67, 5.33] (slightly different from TTEST)

  # Conclusion:
  # TTEST() provides a reasonable estimate of Cohen's d and its 95\% CI,
  # and effectsize::cohens_d() offers another method to compute the CI.
}

}
\seealso{
\code{\link{MANOVA}}, \code{\link{EMMEANS}}
}
