% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/weightwin.R
\name{weightwin}
\alias{weightwin}
\title{Find a weighted climate window}
\usage{
weightwin(Xvar, CDate, BDate, baseline, furthest, closest, FUNC = "L",
  FIXED = FALSE, cutoff.day, cutoff.month, WeightFunction = "W",
  CINTERVAL = "D", par = c(3, 0.2, 0), control = list(ndeps = c(0.01,
  0.01, 0.01)), method = "L-BFGS-B")
}
\arguments{
\item{Xvar}{The climate variable of interest. Please specify the parent
environment and variable name (e.g. Climate$Temp).}

\item{CDate}{The climate date variable. Please specify the parent environment
and variable name (e.g. Climate$Date).}

\item{BDate}{The biological date variable. Please specify the parent
environment and variable name (e.g. Biol$Date).}

\item{baseline}{The baseline model structure used for testing correlation.
Currently known to support lm, lme, glm objects.}

\item{furthest}{The furthest number of days back that you want to search for a
climate window.}

\item{closest}{The closest number of days back that you want any climate
windows to reach.}

\item{FUNC}{The function used to fit the climate variable in the model. Can be
linear ("L"), quadratic ("Q"), cubic ("C"), inverse ("I") or log ("LOG").}

\item{FIXED}{TRUE or FALSE, whether you wish the climate window to be variable
(i.e. the number of days before each biological record is measured) or fixed
(i.e. number of days before a set point in time).}

\item{cutoff.day,cutoff.month}{If FIXED is TRUE, the day and month of the year
from which the fixed window analysis will start.}

\item{WeightFunction}{The distribution to be used for optimisation. Can be
either a Weibull ("W") or Generalised Extreme Value distribution ("G").}

\item{CINTERVAL}{The resolution at which the climate window analysis will be
conducted. May be days ("D"), weeks ("W"), or months ("M"). Note the units
of parameters 'furthest' and 'closest' will differ depending on the choice
of CINTERVAL.}

\item{par}{Shape, scale and location parameters of the Weibull of GEV weight
function used as start weight function. For Weibull : Shape and scale
parameters must be greater than 0, while location parameter must be less
than or equal to 0. For GEV : Scale parameter must be greater than 0.}

\item{control}{Parameters used to determine step size for the optimisation
function. Please see \code{\link{optim}} for more detail.}

\item{method}{The method used for the optimisation function. Please see
\code{\link{optim}} for more detail.}
}
\value{
Produces a constantly updating grid of plots as the optimisation
 function is running. \itemize{ \item Right panel from top to bottom: The
 three parameters (shape, scale and location) determining the weight
 function.

 \item Left top panel: The resulting weight function.

 \item Right middle panel: The delta AICc compared to the baseline model.

 \item Right bottom panel: The weighted mean of climate for the current
 weight function. } Also returns a list containing three objects: \itemize{
 \item BestModel, a model object. The best weighted window model deterimend
 by AICc.

 \item BestModelData, a dataframe. Biological and climate data used to fit
 the best weighted window model.

 \item WeightedOutput, a list. Parameter values for the best weighted window.
 }
}
\description{
Finds the best weighted average of a weather variable over a period that
correlates most strongly with a biological variable. Uses weibull or
Generalised Extreme Value (GEV) distribution. See references for a full
description.
}
\examples{
\dontrun{

 # Test for a weighted average over a fixed climate window
 # using datasets 'Offspring' and 'OffspringClimate'

 # N.B. THIS EXAMPLE MAY TAKE A MOMENT TO CONVERGE ON THE BEST MODEL.

 # Load data

 data(Offspring)
 data(OffspringClimate)

 # Test for climate windows between 365 and 0 days ago (furthest=365, closest=0)
 # Fit a quadratic term for the mean weighted climate (FUNC="Q")
 # in a Poisson regression (offspring number ranges 0-3)
 # Test a variable window (FIXED=FALSE)
 # Test at the resolution of days (CINTERVAL="D")
 # Uses a Weibull weight function (WeightFunction="W")

 weight <- weightwin(Xvar = OffspringClimate$Temperature, CDate = OffspringClimate$Date,
                     BDate = Offspring$Date,
                     baseline = glm(Offspring ~ 1, family = poisson, data = Offspring),
                     furthest = 365, closest = 0, FUNC = "Q",
                     FIXED = FALSE, WeightFunction = "W", CINTERVAL = "D",
                     par = c(3, 0.2, 0), control = list(ndeps = c(0.01, 0.01, 0.01)),
                     method = "L-BFGS-B")

 # View output

 head(weight[[3]])
 summary(weight[[1]])
 head(weight[[2]])
 }
}
\author{
Martijn van de Pol and Liam D. Bailey
}
\references{
van de Pol & Cockburn 2011 Am Nat 177(5):698-707 (doi:
 10.1086/659101) "Identifying the critical climatic time window that affects
 trait expression"
}

