% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_path_pair.R
\name{get_path_pair}
\alias{get_path_pair}
\title{Compute shortest path between origin and destination nodes.}
\usage{
get_path_pair(
  Graph,
  from,
  to,
  algorithm = "bi",
  constant = 1,
  keep = NULL,
  long = FALSE
)
}
\arguments{
\item{Graph}{An object generated by \link{makegraph}, \link{cpp_simplify} or \link{cpp_contract} function.}

\item{from}{A vector of one or more vertices from which shortest paths are calculated (origin).}

\item{to}{A vector of one or more vertices (destination).}

\item{algorithm}{character. \code{Dijkstra} for uni-directional Dijkstra, \code{bi} for bi-directional Dijkstra, \code{A*} for A star unidirectional search or \code{NBA} for New bi-directional A star .Default to \code{bi}}

\item{constant}{numeric. Constant to maintain the heuristic function admissible in A* and NBA algorithms.}

\item{keep}{numeric or character. Vertices of interest that will be returned.}

\item{long}{logical. If \code{TRUE}, a long \code{data.frame} is returned instead of a \code{list}.
Default to 1, when cost is expressed in the same unit than coordinates. See details}
}
\value{
\code{list} or a \code{data.frame} containing shortest path nodes between from and to.
}
\description{
Compute shortest path between origin and destination nodes.
}
\details{
If graph is not contracted, the user has the choice between : \itemize{
  \item unidirectional Dijkstra (\code{Dijkstra})
  \item A star (\code{A*}) : projected coordinates should be provided
  \item bidirectional Dijkstra (\code{bi})
  \item New bi-directional A star (\code{NBA}) : projected coordinates should be provided
}

If the input graph has been contracted by \link{cpp_contract} function, the algorithm is a modified bidirectional search.

In \code{A*} and \code{NBA} algorithms, euclidean distance is used as heuristic function.

All algorithms are \strong{multithreaded.} Please use \code{RcppParallel::setThreadOptions()} to set the number of threads.

To understand the importance of constant parameter, see the package description : \url{https://github.com/vlarmet/cppRouting/blob/master/README.md}
}
\note{
\code{from} and \code{from} must be the same length.
}
\examples{
#Choose number of cores used by cppRouting
RcppParallel::setThreadOptions(numThreads = 1)

#Data describing edges of the graph
edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4),
                  to_vertex=c(1,3,2,4,4,5,1,3,5),
                  cost=c(9,2,11,3,5,12,4,1,6))

#Get all nodes
nodes<-unique(c(edges$from_vertex,edges$to_vertex))

#Construct directed and undirected graph
directed_graph<-makegraph(edges,directed=TRUE)
non_directed<-makegraph(edges,directed=FALSE)

#Sampling origin and destination nodes
origin<-sample(nodes,10,replace=TRUE)
destination<-sample(nodes,10,replace=TRUE)

#Get distance between origin and destination in the two graphs
dir_paths<-get_path_pair(Graph=directed_graph, from=origin, to=destination)
non_dir_paths<-get_path_pair(Graph=non_directed, from=origin, to=destination)
print(dir_paths)
print(non_dir_paths)
}
\seealso{
\link{get_multi_paths}, \link{get_isochrone}, \link{get_detour}
}
