% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/asyncvaried.R
\docType{data}
\name{AsyncVaried}
\alias{AsyncVaried}
\title{Async client for different request types}
\arguments{
\item{..., .list}{Any number of objects of class \code{\link[=HttpRequest]{HttpRequest()}},
must supply inputs to one of these parameters, but not both}
}
\value{
An object of class \code{AsyncVaried} with variables and methods.
Responses are returned in the order they are passed in. We print the
first 10.
}
\description{
Async client for different request types
}
\details{
\strong{Methods}
\describe{
\item{\code{request()}}{
Execute asynchronous requests
- returns: nothing, responses stored inside object,
though will print messages if you choose verbose output
}
\item{\code{requests()}}{
list requests
- returns: a list of \code{HttpRequest} objects, empty list before
requests made
}
\item{\code{responses()}}{
list responses
- returns: a list of \code{HttpResponse} objects, empty list before
requests made
}
\item{\code{parse(encoding = "UTF-8")}}{
parse content
- returns: character vector, empty character vector before
requests made
}
\item{\code{status_code()}}{
(integer) HTTP status codes
- returns: numeric vector, empty numeric vector before
requests made
}
\item{\code{status()}}{
(list) HTTP status objects
- returns: a list of \code{http_code} objects, empty list before
requests made
}
\item{\code{content()}}{
raw content
- returns: raw list, empty list before requests made
}
\item{\code{times()}}{
curl request times
- returns: list of named numeric vectors, empty list before
requests made
}
}
}
\section{Failure behavior}{

HTTP requests mostly fail in ways that you are probably familiar with, 
including when there's a 400 response (the URL not found), and when the 
server made a mistake (a 500 series HTTP status code). 

But requests can fail sometimes where there is no HTTP status code, and 
no agreed upon way to handle it other than to just fail immediately. 

When a request fails when using synchronous requests (see [HttpClient]) 
you get an error message that stops your code progression 
immediately saying for example:

- "Could not resolve host: https://foo.com"
- "Failed to connect to foo.com"
- "Resolving timed out after 10 milliseconds"

However, for async requests we don't want to fail immediately because
that would stop the subsequent requests from occurring. Thus, when 
we find that a request fails for one of the reasons above we 
give back a [HttpResponse] object just like any other response, and:

- capture the error message and put it in the `content` slot of the 
response object (thus calls to `content` and `parse()` work correctly)
- give back a `0` HTTP status code. we handle this specially when testing 
whether the request was successful or not with e.g., the `success()` 
method
}

\examples{
\dontrun{
# pass in requests via ...
req1 <- HttpRequest$new(
  url = "https://httpbin.org/get",
  opts = list(verbose = TRUE),
  headers = list(foo = "bar")
)$get()
req2 <- HttpRequest$new(url = "https://httpbin.org/post")$post()

# Create an AsyncVaried object
out <- AsyncVaried$new(req1, req2)

# before you make requests, the methods return empty objects
out$status()
out$status_code()
out$content()
out$times()
out$parse()
out$responses()

# make requests
out$request()

# access various parts
## http status objects
out$status()
## status codes
out$status_code()
## content (raw data)
out$content()
## times
out$times()
## parsed content
out$parse()
## response objects
out$responses()

# pass in requests in a list via .list param
reqlist <- list(
  HttpRequest$new(url = "https://httpbin.org/get")$get(),
  HttpRequest$new(url = "https://httpbin.org/post")$post(),
  HttpRequest$new(url = "https://httpbin.org/put")$put(),
  HttpRequest$new(url = "https://httpbin.org/delete")$delete(),
  HttpRequest$new(url = "https://httpbin.org/get?g=5")$get(),
  HttpRequest$new(
    url = "https://httpbin.org/post")$post(body = list(y = 9)),
  HttpRequest$new(
    url = "https://httpbin.org/get")$get(query = list(hello = "world"))
)

out <- AsyncVaried$new(.list = reqlist)
out$request()
out$status()
out$status_code()
out$content()
out$times()
out$parse()

# using auth with async
url <- "https://httpbin.org/basic-auth/user/passwd"
auth <- auth(user = "user", pwd = "passwd")
reqlist <- list(
  HttpRequest$new(url = url, auth = auth)$get(),
  HttpRequest$new(url = url, auth = auth)$get(query = list(a=5)),
  HttpRequest$new(url = url, auth = auth)$get(query = list(b=3))
)
out <- AsyncVaried$new(.list = reqlist)
out$request()
out$status()
out$parse()

# failure behavior
## e.g. when a URL doesn't exist, a timeout, etc.
reqlist <- list(
  HttpRequest$new(url = "http://stuffthings.gvb")$get(),
  HttpRequest$new(url = "https://httpbin.org")$head(),
  HttpRequest$new(url = "https://httpbin.org", 
   opts = list(timeout_ms = 10))$head()
)
(tmp <- AsyncVaried$new(.list = reqlist))
tmp$request()
tmp$responses()
tmp$parse("UTF-8")
}
}
\seealso{
Other async: \code{\link{Async}}
}
\keyword{datasets}
