% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/openssl.R
\name{keypair_openssl}
\alias{keypair_openssl}
\title{Asymmetric encryption with openssl}
\usage{
keypair_openssl(
  pub,
  key,
  envelope = TRUE,
  password = NULL,
  authenticated = TRUE
)
}
\arguments{
\item{pub}{An openssl public key.  Usually this will be the path
to the key, in which case it may either the path to a public key
or be the path to a directory containing a file
\code{id_rsa.pub}.  If \code{NULL}, then your public key will be
used (found via the environment variable \code{USER_PUBKEY},
then \code{~/.ssh/id_rsa.pub}).  However, it is not that common
to use your own public key - typically you want either the
sender of a message you are going to decrypt, or the recipient
of a message you want to send.}

\item{key}{An openssl private key.  Usually this will be the path
to the key, in which case it may either the path to a private
key or be the path to a directory containing a file.  You may
specify \code{NULL} here, in which case the environment variable
\code{USER_KEY} is checked and if that is not defined then
\code{~/.ssh/id_rsa} will be used.}

\item{envelope}{A logical indicating if "envelope" encryption
functions should be used.  If so, then we use
\code{openssl::encrypt_envelope} and
\code{openssl::decrypt_envelope}.  If \code{FALSE} then we use
\code{openssl::rsa_encrypt} and \code{openssl::rsa_decrypt}.
See the openssl docs for further details.  The main effect of
this is that using \code{envelope = TRUE} will allow you to
encrypt much larger data than \code{envelope = FALSE}; this is
because openssl asymmetric encryption can only encrypt data up
to the size of the key itself.}

\item{password}{A password for the private key.  If \code{NULL}
then you will be prompted interactively for your password, and
if a string then that string will be used as the password (but
be careful in scripts!)}

\item{authenticated}{Logical, indicating if the result should be
signed with your public key.  If \code{TRUE} then your key will
be verified on decryption.  This provides tampering detection.}
}
\description{
Wrap a pair of openssl keys.  You should pass your private key and
the public key of the person that you are communicating with.
}
\examples{

# Note this uses password = FALSE for use in examples only, but
# this should not be done for any data you actually care about.

# Note that the vignette contains much more information than this
# short example and should be referred to before using these
# functions.

# Generate two keypairs, one for Alice, and one for Bob
path_alice <- tempfile()
path_bob <- tempfile()
cyphr::ssh_keygen(path_alice, password = FALSE)
cyphr::ssh_keygen(path_bob, password = FALSE)

# Alice wants to send Bob a message so she creates a key pair with
# her private key and bob's public key (she does not have bob's
# private key).
pair_alice <- cyphr::keypair_openssl(pub = path_bob, key = path_alice)

# She can then encrypt a secret message:
secret <- cyphr::encrypt_string("hi bob", pair_alice)
secret

# Bob wants to read the message so he creates a key pair using
# Alice's public key and his private key:
pair_bob <- cyphr::keypair_openssl(pub = path_alice, key = path_bob)

cyphr::decrypt_string(secret, pair_bob)

# Clean up
unlink(path_alice, recursive = TRUE)
unlink(path_bob, recursive = TRUE)
}
\seealso{
\code{\link{keypair_sodium}} for a similar function using
  sodium keypairs
}
