#' @title Estimate the Number of True Null Hypotheses
#'
#' @description Estimating the number of true null hypotheses (\code{m0}) 
#' is critical for estiamting the false discover rate (FDR). 
#' The function \code{m0.st} estimates \code{m0} using the method proposed in Storey and Tibshirani (2003). 
#' The function \code{m0.nhcw} estimates \code{m0} using the method proposed in Nettleton et al. (2006).
#' The function \code{m0.bdn} estimates \code{m0} using the method proposed in Bancroft, Du and Nettleton (2013).
#' 
#' @aliases m0.storey m0.nhcw m0.bdn
#' 
#' @return the estimate of \code{m0}.
#' @seealso \code{\link{fdr}} for estimating false discover rate and calculating qvalues.
#' @references 
#' Storey, J. D., & Tibshirani, R. (2003). Statistical significance for genomewide studies. 
#' Proceedings of the National Academy of Sciences, 100(16), 9440-9445.
#'
#' Nettleton, D., Hwang, J. G., Caldo, R. A., & Wise, R. P. (2006). 
#' Estimating the number of true null hypotheses from a histogram of p values. 
#' Journal of agricultural, biological, and environmental statistics, 11(3), 337-356.
#'
#' Bancroft, T., Du, C. and Nettleton, D. (2013), Estimation of False Discovery Rate Using Sequential Permutation p-Values. 
#' Biometrics, 69: 1-7. doi: 10.1111/j.1541-0420.2012.01825.x
#'
#' @examples
#' #simulate 1000 p-vlaues from a mixture beta and uniform distribution. 
#' p = rbunif(1000, alpha=1, beta=29, gamma=0.7)
#' #estimate m0 using the method in Story and Tibshirani (2003).
#' m0.st(p)
#' #estimate m0 using the method in Nettleton et al. (2006)
#' m0.nhcw(p)
#' # ---------- a real example in Bancroft, Du and Nettleton (2013) --------------
#' # read in C++ source code
#' tsf_code = read_code("test_statistics.cpp")
#' # genrate function for sequential permutation test 
#' # for no location difference between two samples
#' if(!exists("cxxspt.mean")){
#'     cxxwrapper(tsf_code, "abs_mean_diff0", "abs_mean_diff", c("n1", "", "int")) -> cxxspt.mean
#' }
#' # download data 
#' if(!file.exists('spt_data.rda')){
#'     download.file('http://dclong.github.io/media/spt/spt_data.rda', 'spt_data.rda')
#' }
#' load('spt_data.rda')
#' cxxspt.mean(leukemia, 10, 1000, n1=5) -> cxxspt.out
#' #estimate m0 using the method in Bancroft, Du and Nettleton (2013).
#' m0.bdn(cxxspt.out, 20) 
#' 
#' @rdname est_m0
#' @export
#' @param p a vector of pvalues.
#' @param lambda points chosen for fitting spline which can be considered 
#' as tuning parameters in estimating m0. 
#' These points must be in [0,1] and a defualt value seq(0,0.95,0.05) is used. 
#' For more information, please see Storey and Tibshirani (2003).

m0.st = function(p,lambda=seq(0,0.95,0.05)){
  m = length(p) 
  lambda.len = length(lambda)
  pi0 = rep(0,lambda.len)
  for(i in 1:lambda.len) {
    pi0[i] <- mean(p>=lambda[i]) / (1-lambda[i])
  }
  spi0 <- smooth.spline(lambda,pi0,df=3)
  pi0 <- max(predict(spi0,x=1)$y,0)
  pi0 <- min(pi0,1)
  return(pi0*m)
} 

#' @rdname est_m0
#' @param bins number of bins to be used in the histogram-based methods.
#' A default value 20 is used, which is recommended in Netlleton et al. (2006)
#' and Bancroft, Du and Nettleton (2013).
#' @export
m0.nhcw = function(p, bins=20) {
    m <- length(p)
    index = 1:bins
    bin <- c(-0.1, index/bins)
    bin[bins+1] = 2#make the last cut-off one bigger to avoid numerical problems
    bin.counts=rep(0,bins)
    for(i in index){
        bin.counts[i]=sum((p>bin[i])&(p<=bin[i+1]))
    }
    tail.means <- rev(cumsum(rev(bin.counts))/index)
    tail.means[min(which(bin.counts - tail.means <= 0))] * bins
}

#' @rdname est_m0
#' @param obj an object of class "spt" (returned by \code{spt} or wrapper function generated by \code{cxxwrapper}).
#' @export
m0.bdn = function(obj, bins=20){
    delta = obj$h / (10 * obj$n^2)
    cutoff = bcutoff(obj$p, obj$h, obj$n, delta, bins)
    probs = cutoff[,3]
    counts = cutoff[,4]
    revCumSum.prob = rev(cumsum(rev(probs)))
    revCumSum.count = rev(cumsum(rev(counts)))
    fraction.prob = probs/revCumSum.prob
    fraction.count = counts/revCumSum.count
    flag = min(which(fraction.count<=fraction.prob))
    revCumSum.count[flag]/revCumSum.prob[flag]
}

#' @noRd
#' Calculate cut-offs for estimating m0 based on non-uniformly distributed
#' pvalues under null hypotheses. The cut-offs are decided based on the 
#' rule of making approximately equl probability for each bin. 
#' For more detains, see Bancroft et al. (2013). 
bcutoff = function(p, h, n, delta, bins){
  prob = 1/bins
  pvals.support(h,n) -> pvals.theory
  pvals = pvals.theory[,1]
  cutoff = NULL
  currentCut = 0
  while(currentCut<1-delta){
    index = which(pvals>currentCut+prob-delta)
    if(length(index)==0){
      currentCut = 1
    }else{
      currentCut = min(pvals[index])
    }
    if(currentCut<1-delta){
      cutoff = c(cutoff,currentCut)
    }else{
      cutoff = c(cutoff,1)
    }
  }
  cutoff.length = length(cutoff)
  cutoff = cbind(rep(0,cutoff.length),cutoff,rep(0,cutoff.length),rep(0,cutoff.length))
  for(i in 2:cutoff.length){
    cutoff[i,1] = cutoff[i-1,2]
  }
  for(i in 1:cutoff.length){
    cutoff[i,3] = cutoff[i,2] - cutoff[i,1]
    cutoff[i,4] = sum(p>cutoff[i,1]+delta&p<cutoff[i,2]+delta)
  }
  colnames(cutoff) = c("lower","upper","prob","count")
  cutoff
}

#' @noRd
#' Calculate theorectical sequential pvalues and corresponding probabilities.
pvals.support= function(h,n){
  pvals = c((1:(h-1))/n,h/(n:h))
  prob = diff(c(0,pvals))
  cbind(pvals,prob)
}
