% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictratingdist_DDMConf.R
\name{predictDDMConf}
\alias{predictDDMConf}
\alias{predictDDMConf_Conf}
\alias{predictDDMConf_RT}
\title{Prediction of Confidence Rating and Reaction Time Distribution in the drift diffusion confidence model}
\usage{
predictDDMConf_Conf(paramDf, maxrt = 15, subdivisions = 100L,
  stop.on.error = FALSE, .progress = TRUE)

predictDDMConf_RT(paramDf, maxrt = 9, subdivisions = 100L, minrt = NULL,
  scaled = FALSE, DistConf = NULL, .progress = TRUE)
}
\arguments{
\item{paramDf}{a list or data frame with one row. Column names should match the names of
\link{DDMConf} model parameter names. For different stimulus quality/mean
drift rates, names should be \code{v1}, \code{v2}, \code{v3},....
Different \code{sv} and/or \code{s} parameters are possible with \code{sv1}, \code{sv2}, \code{sv3}... (\code{s1}, \code{s2}, \code{s3},...
respectively) with equally many steps as for drift rates. Additionally, the confidence
thresholds should be given by names with \code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},....}

\item{maxrt}{numeric. The maximum RT for the
integration/density computation. Default: 15 (for \code{predictDDMConf_Conf}
(integration)), 9 (for \code{predictDDMConf_RT}).}

\item{subdivisions}{\code{integer} (default: 100).
For \code{predictDDMConf_Conf} it is used as argument for the inner integral routine.
For \code{predictDDMConf_RT} it is the number of points for which the density is computed.}

\item{stop.on.error}{logical. Argument directly passed on to integrate. Default is \code{FALSE},
since the densities invoked may lead to slow convergence of the integrals (which are still
quite accurate) which causes R to throw an error.}

\item{.progress}{logical. If \code{TRUE} (default) a progress bar is drawn to the console.}

\item{minrt}{numeric or \code{NULL}(default). The minimum rt for the density computation.}

\item{scaled}{logical. For \code{predictDDMConf_RT}. Whether the computed density
should be scaled to integrate to one (additional column \code{densscaled}). Otherwise the output
contains only the defective density (i.e. its integral is equal to the probability of a
response and not 1). If \code{TRUE}, the argument \code{DistConf} should be given, if available.
Default: \code{FALSE}.}

\item{DistConf}{\code{NULL} or \code{data.frame}. A \code{data.frame} or \code{matrix} with column
names, giving the distribution of response and rating choices for
different conditions and stimulus categories in the form of the output of
\code{predictDDMConf_Conf}. It is only necessary, if \code{scaled=TRUE}, because these
probabilities are used for scaling. If \code{scaled=TRUE} and \code{DistConf=NULL}, it will be computed
with the function \code{predictDDMConf_Conf}, which takes some time and the function will
throw a message. Default: \code{NULL}}
}
\value{
\code{predictDDMConf_Conf} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{p}, \code{info}, \code{err}. \code{p} is the predicted probability of a response
and \code{rating}, given the stimulus category and condition. \code{info} and \code{err} refer to the
respective outputs of the integration routine used for the computation.
\code{predictDDMConf_RT} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{rt} and \code{dens} (and \code{densscaled}, if \code{scaled=TRUE}).
}
\description{
\code{predictDDMConf_Conf} predicts the categorical response distribution of
decision and confidence ratings, \code{predictDDMConf_RT} computes the
RT distribution (density) in the drift diffusion confidence model
(Hellmann et al., preprint), given specific parameter
constellations. See \code{\link{dDDMConf}} for more information about the model
and parameters.
}
\details{
The function \code{predictDDMConf_Conf} consists merely of an integration of
the response time density, \code{\link{dDDMConf}}, over the
response time in a reasonable interval (0 to \code{maxrt}). The function
\code{predictDDMConf_RT} wraps these density
functions to a parameter set input and a \code{data.frame} output.
For the argument \code{paramDf}, the output of the fitting function \code{\link{fitRTConf}}
with the DDMConf model may be used.
}
\note{
Different parameters for different conditions are only allowed for drift rate
\code{v}, drift rate variability \code{sv}, and process variability \code{s}. Otherwise, \code{s} is
not required in \code{paramDf} but set to 1 by default. All other parameters are used for all
conditions.
}
\examples{
# 1. Define some parameter set in a data.frame
paramDf <- data.frame(a=2,v1=0.5, v2=1, t0=0.1,z=0.55,
                      sz=0,sv=0.2, st0=0, theta1=0.8)

# 2. Predict discrete Choice x Confidence distribution:
preds_Conf <- predictDDMConf_Conf(paramDf,  maxrt = 15)
head(preds_Conf)

# 3. Compute RT density
preds_RT <- predictDDMConf_RT(paramDf, maxrt=4, subdivisions=200) #(scaled=FALSE)
# same output with scaled density column:
preds_RT <- predictDDMConf_RT(paramDf, maxrt=4, subdivisions=200,
                              scaled=TRUE, DistConf = preds_Conf)
head(preds_RT)

\donttest{
  # Example of visualization
  library(ggplot2)
  preds_Conf$rating <- factor(preds_Conf$rating, labels=c("unsure", "sure"))
  preds_RT$rating <- factor(preds_RT$rating, labels=c("unsure", "sure"))
  ggplot(preds_Conf, aes(x=interaction(rating, response), y=p))+
    geom_bar(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")
  ggplot(preds_RT, aes(x=rt, color=interaction(rating, response), y=dens))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")+
    theme(legend.position = "bottom")
  ggplot(aggregate(densscaled~rt+correct+rating+condition, preds_RT, mean),
         aes(x=rt, color=rating, y=densscaled))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(condition), rows=vars(correct), labeller = "label_both")+
    theme(legend.position = "bottom")
}
# Use PDFtoQuantiles to get predicted RT quantiles
head(PDFtoQuantiles(preds_RT, scaled = FALSE))

}
\author{
Sebastian Hellmann.
}
