% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_edr.R
\name{plot_edr}
\alias{plot_edr}
\title{Plot Ecological Dynamic Regimes}
\usage{
plot_edr(
  x,
  trajectories,
  states,
  traj.colors = NULL,
  state.colors = NULL,
  variable = NULL,
  type = "trajectories",
  axes = c(1, 2),
  initial = F,
  ...
)
}
\arguments{
\item{x}{Symmetric matrix or \code{dist} object containing the dissimilarities
between each pair of states of all trajectories in the EDR. Alternatively, data
frame containing the coordinates of all trajectory states in an ordination space.}

\item{trajectories}{Vector indicating the trajectory or site to which each
state in \code{x} belongs.}

\item{states}{Vector of integers indicating the order of the states in \code{x} for
each trajectory.}

\item{traj.colors}{Specification for the color of all individual trajectories
(defaults "grey") or a vector with length equal to the number of different
trajectories indicating the color for each individual trajectory.}

\item{state.colors}{Specification for the color of all trajectory states
(defaults equal to \code{traj.colors}), vector with length equal to the number
of states indicating the color for each trajectory state, or vector of colors
used to generate a gradient depending on the values of \code{variable} (if
\code{type = "gradient"}).}

\item{variable}{Numeric vector with equal length to the number of states to
be represented using a gradient of state colors (if \code{type = "gradient"}).}

\item{type}{One of the following \code{"trajectories"}, \code{"states"}, or \code{"gradient"}.}

\item{axes}{An integer vector indicating the pair of axes in the ordination
space to be plotted.}

\item{initial}{Flag indicating if the initial state must be plotted (only if
\code{type = "states"} or \code{type = "gradient"})}

\item{...}{Arguments for generic \code{\link[=plot]{plot()}}.}
}
\value{
\code{plot_edr()} permits representing the trajectories of an Ecological Dynamic
Regime using different colors for each trajectory or state.
}
\description{
Represents EDR trajectories in the state space. Trajectories and/or states can
be displayed in different colors based in a predefined classification or variable.
}
\examples{

# Data
state_variables <- EDR_data$EDR1$abundance
d <- EDR_data$EDR1$state_dissim

# Coordinates in classic multidimensional scaling
x <- cmdscale(d, k = 3)

# Plot trajectories 1-10 in "coral", 11-20 in "blue" and 21-30 in "gold"
plot_edr(x = x, trajectories = state_variables$traj,
         states = as.integer(state_variables$state),
         traj.colors = c(rep("coral", 10), rep("royalblue", 10), rep("gold", 10)),
         main = "type = 'trajectories'")
legend("bottomleft", legend = paste0("Trajectories ", c("1-10", "11-20", "21-30")),
       lty = 1, col = c("coral", "royalblue", "gold"))

# Plot states with different colors depending on the state value
plot_edr(x = x, trajectories = state_variables$traj,
         states = as.integer(state_variables$state),
         traj.colors = NULL,
         state.colors = rep(RColorBrewer::brewer.pal(5, "Blues"),
                            length(unique(state_variables$traj))),
         type = "states", main = "type = 'states'")
legend("bottomleft", legend = paste0("State ", 1:5),
       pch = 15, col = RColorBrewer::brewer.pal(5, "Blues"))

# Plot states with different colors depending on the abundance of sp1
plot_edr(x = x, trajectories = state_variables$traj,
         states = as.integer(state_variables$state),
         traj.colors = NULL, state.colors = viridis::viridis(5),
         variable = state_variables$sp1,
         type = "gradient", main = "type = 'gradient'", initial = TRUE)
legend("bottomleft",
       legend = c(paste0("abun sp1 = ", min(state_variables$sp1)),
                  rep(NA, 28),
                  paste0("abun sp1 = ", max(state_variables$sp1))),
       fill = viridis::viridis(30), border = NA, y.intersp = 0.2)
}
\seealso{
\code{\link[=plot.RETRA]{plot.RETRA()}} for plotting representative trajectories in an ordination space
representing the state space of the EDR.
}
\author{
Martina Sánchez-Pinillos
}
