##' Get trackdata from loaded emuDB
##' 
##' Extract trackdata information from a loaded emuDB that 
##' corresponds to the entries in a segment list.
##' 
##' This function utilizes the wrassp package for signal processing and 
##' SSFF/audio file handling. It reads time relevant data from a given 
##' segment list (\code{\link{emuRsegs}} or \code{\link{emusegs}}), extracts the 
##' specified trackdata and places it into a 
##' trackdata object (analogous to the depricated \code{emu.track}). This function
##' replaces the deprecated \code{emu.track} function. Note that a error is thrown
##' if the bundles in the \code{\link{emuRsegs}} or \code{\link{emusegs}} object 
##' have in-homogeneous sampling rates as this will lead to inconsistent/erroneous
##' \code{\link{trackdata}} or \code{\link{emuRtrackdata}} result objects. For 
##' more information on the structural elements of an emuDB 
##' see \code{vignette{emuDB}}.
##' 
##' @param emuDBhandle emuDB handle as returned by \code{\link{load_emuDB}}
##' @param seglist \code{\link{emuRsegs}} or \code{\link{emusegs}} object obtained by \code{\link{query}}ing a loaded emuDB 
##' @param ssffTrackName The name of track that one wishes to extract (see 
##' \code{\link{list_ssffTrackDefinitions}} for the defined ssffTracks of the 
##' emuDB). If the parameter \code{onTheFlyFunctionName} is set, then 
##' this corresponds to the column name af the AsspDataObj (see
##' \code{wrasspOutputInfos[[onTheFlyFunctionName]]$tracks} and \code{\link{wrasspOutputInfos}}). 
##' If the parameter \code{onTheFlyFunctionName} is set and this one isn't, then per default
##' the first track listed in the \code{wrasspOutputInfos} is chosen (\code{wrasspOutputInfos[[onTheFlyFunctionName]]$tracks[1]}).
##' @param cut An optional cut time for segment data, ranges between 
##' 0 and 1, a value of 0.5 will extract data only at the segment midpoint.
##' @param npoints An optional number of points to retrieve for each segment or event. 
##' For segments this requires the \code{cut} parameter to be set; if this is the case, then data is extracted around the resulting cut time. 
##' For events data is extracted around the event time. If npoints is an odd number, the 
##' samples are centered around the cut-time-sample; if not, they are skewed to the
##' right by one sample.
##' @param onTheFlyFunctionName Name of wrassp function that will perform the on-the-fly 
##' calculation (see \code{names(wrasspOutputInfos)} for a list of all the signal processing functions wrassp provides)
##' @param onTheFlyParams A \code{pairlist} of parameters that will be given to the function 
##' passed in by the \code{onTheFlyFunctionName} parameter. This list can easily be 
##' generated by applying the \code{formals} function to the on-the-fly function name and then setting the according 
##' parameter one wishes to change.     
##' @param onTheFlyOptLogFilePath Path to optional log file for on-the-fly function
##' @param nrOfAllocationRows If this size limit of the data matrix is reached 
##' a further \code{nrOfAllocationRows} more rows will be allocated. As this allocation leads to
##' a performance penalty one should consider increasing this number for large emuDBs. 
##' @param resultType Specify class of returned object. Either \code{"emuRtrackdata"} or \code{"trackdata"}.
##' @param verbose Show progress bars and further information
##' @return If the \code{cut} parameter is not set (the default) an object of type \code{\link{trackdata}} or \code{\link{emuRtrackdata}} 
##' is returned. If \code{cut} is set and \code{npoints} is not, or the seglist 
##' is of type event and npoints is not set, a \code{\link{data.frame}} is returned.
##' @seealso \code{\link{formals}}, \code{\link{wrasspOutputInfos}}, \code{\link{trackdata}}, \code{\link{emuRtrackdata}}
##' @keywords misc
##' @import wrassp
##' @aliases emu.track
##' @export
##' @examples
##' \dontrun{
##' 
##' ##################################
##' # prerequisite: loaded ae emuDB 
##' # (see ?load_emuDB for more information)
##' 
##' # query loaded "ae" emuDB for all "i:" segments of the "Phonetic" level
##' sl = query(emuDBhandle = ae, 
##'            query = "Phonetic == i:")
##' 
##' # get the corresponding formant trackdata
##' td = get_trackdata(emuDBhandle = ae, 
##'                    seglist = sl, 
##'                    ssffTrackName = "fm")
##' 
##' # get the corresponding F0 trackdata
##' # as there is no F0 ssffTrack defined in the "ae" emuDB we will 
##' # calculate the necessary values on-the-fly
##' td = get_trackdata(emuDBhandle = ae, 
##'                    seglist = sl, 
##'                    onTheFlyFunctionName = "ksvF0")
##'                    
##' }
##' 
"get_trackdata" <- function(emuDBhandle, seglist = NULL, ssffTrackName = NULL, cut = NULL, 
                            npoints = NULL, onTheFlyFunctionName = NULL, onTheFlyParams = NULL, 
                            onTheFlyOptLogFilePath = NULL, nrOfAllocationRows = 10000, 
                            resultType = "trackdata", verbose = TRUE){
  #########################
  # get DBconfig
  DBconfig = load_DBconfig(emuDBhandle)
  
  #########################
  # parameter checks  
  
  # set ssffTrackName to fist tracks entry in wrasspOutputInfos if ssffTrackName is not set
  if(!is.null(onTheFlyFunctionName) && is.null(ssffTrackName)){
    ssffTrackName = wrassp::wrasspOutputInfos[[onTheFlyFunctionName]]$tracks[1]
  }
  
  # check if all values for minimal call are set
  if( is.null(emuDBhandle) || is.null(seglist) || is.null(ssffTrackName)) {
    stop("emuDBhandle, seglist and ssffTrackName have to all be set!\n")
  }
  
  # check if cut value is correct
  if(!is.null(cut)){
    if(cut < 0 || cut > 1){
      stop('Bad value given for cut argument. Cut can only be a value between 0 and 1!')
    }
    if(emusegs.type(seglist) == 'event'){
      stop("Cut value should not be set if emusegs.type(Seglist) == 'event'!")
    }
  }
  
  # check if npoints value is correct
  if(!is.null(npoints)){
    if(is.null(cut) && emusegs.type(seglist) != 'event'){
      stop('Cut argument hast to be set or seglist has to be of type event if npoints argument is used.')
    }
  }
  
  # check if onTheFlyFunctionName is set if onTheFlyParams is
  if(is.null(onTheFlyFunctionName) && !is.null(onTheFlyParams)){
    stop('onTheFlyFunctionName has to be set if onTheFlyParams is set!')
  }
  
  # check if both onTheFlyFunctionName and onTheFlyParams are set if onTheFlyOptLogFilePath is 
  if( !is.null(onTheFlyOptLogFilePath) && (is.null(onTheFlyFunctionName) || is.null(onTheFlyParams))){
    stop('Both onTheFlyFunctionName and onTheFlyParams have to be set for you to be able to use the onTheFlyOptLogFilePath parameter!')
  }
  
  # check resultType if valid string
  if(!resultType %in% c("emuRtrackdata", "trackdata")){
    stop("resultType has to either be 'emuRtrackdata' or 'trackdata'")
  }
  
  if(resultType == "emuRtrackdata" && class(seglist) =="emusegs"){
    stop("resultType can only be 'trackdata', if a seglist of class 'emusegs' is passed in")
  }
  
  
  #########################
  # get track definition
  if(is.null(onTheFlyFunctionName)){
    trackDefFound = sapply(DBconfig$ssffTrackDefinitions, function(x){ x$name == ssffTrackName})
    trackDef = DBconfig$ssffTrackDefinitions[trackDefFound]
    
    # check if correct nr of trackDefs where found
    if(length(trackDef) != 1){
      if(length(trackDef) < 1 ){
        stop('The emuDB object ', DBconfig$name, ' does not have any ssffTrackDefinitions called ', ssffTrackName)
      }else{
        stop('The emuDB object ', DBconfig$name, ' has multiple ssffTrackDefinitions called ', ssffTrackName, '! This means the DB has an invalid _DBconfig.json')
      }
    }
  }else{
    trackDef = list()
    trackDef[[1]] = list()
    trackDef[[1]]$name = ssffTrackName
    trackDef[[1]]$columnName =  ssffTrackName
  }
  
  ###################################
  # check for sample rate consistancy
  uniqSessionBndls =utils::read.table(text = as.character(dplyr::distinct_(seglist, "utts")$utts), sep = ":", 
                                      col.names = c("session", "bundle"), colClasses = c("character", "character"), stringsAsFactors = F)
  DBI::dbGetQuery(emuDBhandle$connection,"CREATE TEMP TABLE uniq_session_bndls_tmp (session TEXT,bundle TEXT)")
  DBI::dbWriteTable(emuDBhandle$connection, "uniq_session_bndls_tmp", uniqSessionBndls, append = T)
  sesBndls = DBI::dbGetQuery(emuDBhandle$connection, paste0("SELECT bundle.db_uuid, bundle.session, bundle.name, bundle.annotates, bundle.sample_rate, bundle.md5_annot_json ",
                                                            "FROM uniq_session_bndls_tmp, bundle ",
                                                            "WHERE uniq_session_bndls_tmp.session = bundle.session AND uniq_session_bndls_tmp.bundle = bundle.name"))
  DBI::dbGetQuery(emuDBhandle$connection,"DROP TABLE uniq_session_bndls_tmp")
  
  # remove uuid & MD5sum because we don't want to scare our users :-)
  sesBndls$db_uuid = NULL
  sesBndls$MD5annotJSON = NULL
  if(length(unique(sesBndls$sample_rate)) != 1){
    stop("The emusegs/emuRsegs object passed in refers to bundles with in-homogeneous sampling rates in their audio files! Here is a list of all refered to bundles incl. their sampling rate: \n", paste(utils::capture.output(print(sesBndls)), collapse = "\n"))
  }
  
  ###################################
  #create empty index, ftime matrices
  index <- matrix(ncol=2, nrow=length(seglist$utts))
  colnames(index) <- c("start","end")
  
  ftime <- matrix(ncol=2, nrow=length(seglist$utts))
  colnames(ftime) <- c("start","end")
  
  data <- NULL
  origFreq <- NULL
  
  
  ########################
  # preallocate data (needs first element to be read)
  
  
  splUtt = stringr::str_split(seglist$utts[1], ':')[[1]]
  
  # check if utts entry exists
  bndls = list_bundles(emuDBhandle)
  if(!any(bndls$session == splUtt[1] & bndls$name == splUtt[2])){
    stop("Following utts entry not found: ", seglist$utts[1])
  }
  
  
  if(!is.null(onTheFlyFunctionName)){
    funcFormals = NULL
    qr = DBI::dbGetQuery(emuDBhandle$connection, paste0("SELECT * FROM bundle WHERE db_uuid='", emuDBhandle$UUID, "' AND session='",
                                                        splUtt[1], "' AND name='", splUtt[2], "'"))
    funcFormals$listOfFiles = file.path(emuDBhandle$basePath, paste0(qr$session, session.suffix), paste0(qr$name, bundle.dir.suffix), qr$annotates)
    funcFormals$toFile = FALSE
    curDObj = do.call(onTheFlyFunctionName,funcFormals)
  }else{
    fpath <- file.path(emuDBhandle$basePath, paste0(splUtt[1], session.suffix), paste0(splUtt[2], bundle.dir.suffix), paste0(splUtt[2], ".", trackDef[[1]]$fileExtension))
    curDObj <- wrassp::read.AsspDataObj(fpath)
  }
  tmpData <- eval(parse(text = paste("curDObj$", trackDef[[1]]$columnName, sep = "")))
  if(verbose){
    cat('\n  INFO: preallocating data matrix with:', ncol(tmpData), ',', nrOfAllocationRows, 
        'columns and rows.')
  }
  data <- matrix(ncol = ncol(tmpData), nrow = nrOfAllocationRows) # preallocate
  timeStampRowNames = numeric(nrOfAllocationRows) - 1 # preallocate rownames vector. -1 to set default val other than 0
  
  
  
  
  ###############################
  # set up function formals + pb
  if(!is.null(onTheFlyFunctionName)){
    funcFormals = formals(onTheFlyFunctionName)
    funcFormals[names(onTheFlyParams)] = onTheFlyParams
    funcFormals$toFile = FALSE
    funcFormals$optLogFilePath = onTheFlyOptLogFilePath
    if(verbose){
      cat('\n  INFO: applying', onTheFlyFunctionName, 'to', length(seglist$utts), 'files\n')
      pb <- utils::txtProgressBar(min = 0, max = length(seglist$utts), style = 3)
    }
  }else{
    if(verbose){
      cat('\n  INFO: parsing', length(seglist$utts), trackDef[[1]]$fileExtension, 'files\n')
      pb <- utils::txtProgressBar(min = 0, max = length(seglist$utts), style = 3)
    }
  }
  
  prevUtt = ""
  
  # loop through bundle names
  curIndexStart = 1
  for (i in 1:length(seglist$utts)){
    curUtt = seglist$utts[i]
    splUtt = stringr::str_split(curUtt, ':')[[1]]
    
    # check if utts entry exists
    bndls = list_bundles(emuDBhandle)
    if(!any(bndls$session == splUtt[1] & bndls$name == splUtt[2])){
      stop("Following utts entry not found: ", seglist$utts[i])
    }
    
    fpath <- file.path(emuDBhandle$basePath, paste0(splUtt[1], session.suffix), paste0(splUtt[2], bundle.dir.suffix), paste0(splUtt[2], ".", trackDef[[1]]$fileExtension))
    
    ################
    #get data object
    
    if(!is.null(onTheFlyFunctionName)){
      qr = DBI::dbGetQuery(emuDBhandle$connection, paste0("SELECT * FROM bundle WHERE db_uuid='", emuDBhandle$UUID, "' AND session='",
                                                          splUtt[1], "' AND name='", splUtt[2], "'"))
      funcFormals$listOfFiles = file.path(emuDBhandle$basePath, paste0(qr$session, session.suffix), paste0(qr$name, bundle.dir.suffix), qr$annotates)
      # only perform calculation if curUtt is not equal to preUtt
      if(curUtt != prevUtt){
        curDObj = do.call(onTheFlyFunctionName, funcFormals)
      }
      
      if(verbose){
        utils::setTxtProgressBar(pb, i)
      }
    }else{
      curDObj <- wrassp::read.AsspDataObj(fpath)
      if(verbose){
        utils::setTxtProgressBar(pb, i)
      }
    }
    
    # set origFreq 
    origFreq <- attr(curDObj, "origFreq")
    
    # set curStart+curEnd
    curStart <- seglist$start[i]
    if(emusegs.type(seglist) == 'event'){
      curEnd <- seglist$start[i]
    }else{
      curEnd <- seglist$end[i]
    }
    
    
    fSampleRateInMS <- (1 / attr(curDObj, "sampleRate")) * 1000
    fStartTime <- attr(curDObj, "startTime") * 1000
    
    # add one on if event to be able to capture in breakValues 
    if(emusegs.type(seglist) == 'event'){
      if(npoints == 1 || is.null(npoints)){
        timeStampSeq <- seq(fStartTime, curEnd + fSampleRateInMS, fSampleRateInMS)
      }else{
        timeStampSeq <- seq(fStartTime, curEnd + fSampleRateInMS * npoints, fSampleRateInMS)
      }
    }else{
      if(npoints == 1 || is.null(npoints)){
        timeStampSeq <- seq(fStartTime, curEnd, fSampleRateInMS)
      }else{
        timeStampSeq <- seq(fStartTime, curEnd + fSampleRateInMS * npoints, fSampleRateInMS)
      }
    }
    
    ##################################################
    # search for first element larger than start time
    breakVal <- -1
    for (j in 1:length(timeStampSeq)){
      if (timeStampSeq[j] >= curStart){
        breakVal <- j
        break
      }
    }
    # check if breakVal was found
    if(breakVal == -1){
      stop("No track samples found belonging to sl_rowIdx: ", i, " with values: ", paste0(seglist[i,], collapse = " "), 
           "! This is probably due to a very short SEGMENT that doesn't contain any '", ssffTrackName, "' values.")
    }
    
    curStartDataIdx <- breakVal
    curEndDataIdx <- length(timeStampSeq)
    
    
    ################
    # extract data
    tmpData <- eval(parse(text = paste("curDObj$", trackDef[[1]]$columnName, sep = "")))
    
    #############################################################
    # set curIndexEnd dependant on if event/segment/cut/npoints
    if(!is.null(cut) || emusegs.type(seglist) == 'event'){
      if(emusegs.type(seglist) == 'event'){
        cutTime = curStart
        curEndDataIdx <- curStartDataIdx
        curStartDataIdx = curStartDataIdx - 1 # last to elements are relevant -> move start to left
      }else{
        cutTime = curStart + (curEnd - curStart) * cut
      }
      
      sampleTimes = timeStampSeq[curStartDataIdx:curEndDataIdx]
      closestIdx = which.min(abs(sampleTimes - cutTime))
      cutTimeSampleIdx = curStartDataIdx + closestIdx - 1
      
      if(is.null(npoints) || npoints == 1){
        # reset data idxs
        curStartDataIdx = curStartDataIdx + closestIdx - 1
        curEndDataIdx = curStartDataIdx
        curIndexEnd = curIndexStart
      }else{
        # reset data idx
        halfNpoints = (npoints - 1) / 2 # -1 removes cutTimeSample
        curStartDataIdx = cutTimeSampleIdx - floor(halfNpoints)
        curEndDataIdx = cutTimeSampleIdx + ceiling(halfNpoints)
        curIndexEnd = curIndexStart + npoints - 1
      }
      
    }else{
      # normal segments
      curIndexEnd <- curIndexStart + curEndDataIdx - curStartDataIdx
    }
    # set index and ftime
    index[i,] <- c(curIndexStart, curIndexEnd)
    ftime[i,] <- c(timeStampSeq[curStartDataIdx], timeStampSeq[curEndDataIdx])
    
    #############################
    # calculate size of and create new data matrix
    rowSeq <- seq(timeStampSeq[curStartDataIdx], timeStampSeq[curEndDataIdx], fSampleRateInMS) 
    curData <- matrix(ncol = ncol(tmpData), nrow = length(rowSeq))
    
    # check if it is possible to extract curData 
    if(curStartDataIdx > 0 && curEndDataIdx <= dim(tmpData)[1]){
      curData[,] <- tmpData[curStartDataIdx:curEndDataIdx,]
    }else{
      entry= paste(seglist[i,], collapse = " ")
      stop('Can not extract following segmentlist entry: ', entry, ' start and/or end times out of bounds')
    }
    
    ##############################
    # Check if enough space (expand data matrix if necessary) 
    # then append to data matrix 
    if(dim(data)[1]<curIndexEnd){
      if(verbose){
        cat('\n  INFO: allocating more space in data matrix')
      }
      data = rbind(data, matrix(ncol = ncol(data), nrow = nrOfAllocationRows))
      timeStampRowNames = c(timeStampRowNames, numeric(nrOfAllocationRows) - 1)
    }
    
    data[curIndexStart:curIndexEnd,] = curData
    timeStampRowNames[curIndexStart:curIndexEnd] <- rowSeq
    curIndexStart <- curIndexEnd + 1
    
    prevUtt = curUtt
    
  }
  
  ########################################
  # remove superfluous NA vals from data
  if(verbose){
    cat('\n  INFO: removing superfluous NA vals from over-allocated data matrix\n')
  }
  data = data[complete.cases(data),]
  data = as.matrix(data) # make sure it is a matrix to be able to set row names
  timeStampRowNames = timeStampRowNames[timeStampRowNames != -1]
  
  if((!is.null(cut) && (npoints == 1 || is.null(npoints))) || (emusegs.type(seglist) == 'event' && (npoints == 1 || is.null(npoints)))){
    resObj = as.data.frame(data)
    colnames(resObj) = paste(trackDef[[1]]$columnName, seq(1:ncol(resObj)), sep = '')    
  }else{
    rownames(data) <- timeStampRowNames
    colnames(data) <- paste("T", 1:ncol(data), sep = "")
    ########################################
    #convert data, index, ftime to trackdata
    resObj <- as.trackdata(data, index=index, ftime, ssffTrackName)
    
    if(any(trackDef[[1]]$columnName %in% c("dft", "css", "lps", "cep"))){
      if(!is.null(origFreq)){
        if(verbose){
          cat('\n  INFO: adding fs attribute to trackdata$data fields')
        }
        attr(resObj$data, "fs") <- seq(0, origFreq/2, length=ncol(resObj$data))
        class(resObj$data) <- c(class(resObj$data), "spectral")
      }else{
        stop("no origFreq entry in spectral data file!")
      }
    }
  }
  
  # close progress bar if open
  if(exists('pb')){
    close(pb)
  }
  
  # convert to emuRtrackdata if resultType is 'emuRtrackdata'
  if(resultType =="emuRtrackdata"){
    resObj = create_emuRtrackdata(seglist, resObj)
  }
  
  return(resObj)
  
}

#######################
# FOR DEVELOPMENT
# library('testthat')
# test_file('tests/testthat/test_emuR-get_trackdata.R')
