% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evalThreshold.r
\name{evalThreshold}
\alias{evalThreshold}
\title{Weighted thresholds for predictions}
\usage{
evalThreshold(
  pres,
  contrast,
  presWeight = rep(1, length(pres)),
  contrastWeight = rep(1, length(contrast)),
  at = c("msss", "mdss", "minPres", "prevalence", "sensitivity"),
  sensitivity = 0.9,
  thresholds = seq(0, 1, by = 0.001),
  na.rm = FALSE,
  ...
)
}
\arguments{
\item{pres}{Numeric vector. Predicted values at test presences.}

\item{contrast}{Numeric vector. Predicted values at background/absence sites.}

\item{presWeight}{Numeric vector same length as \code{pres}. Relative weights of presence sites. The default is to assign each presence a weight of 1.}

\item{contrastWeight}{Numeric vector same length as \code{contrast}. Relative weights of background sites. The default is to assign each presence a weight of 1.}

\item{at}{Character or character vector, name(s) of threshold(s) to calculate. The default is to calculate them all.
\itemize{
        \item \code{'msss'}: Threshold that the maximizes the sum of sensitivity and specificity.
        \item \code{'mdss'}: Threshold that minimizes the difference between sensitivity and specificity.
        \item \code{'minPres'}: Minimum prediction across presences. This threshold is not weighted.
        \item \code{'prevalence'}: Prevalence of presences (sum(presence weights) / sum(presence weights + background weights))'
        \item \code{'sensitivity'}: Threshold that most closely returns the sensitivity specified by \code{sensitivity}.
}}

\item{sensitivity}{Value of specificity to match (used only if \code{at} contains \code{'sensitivity'}).}

\item{thresholds}{Numeric vector. Thresholds at which to calculate the sum of sensitivity and specificity. The default evaluates all values from 0 to 1 in steps of 0.01.}

\item{na.rm}{Logical. If \code{TRUE} then remove any presences and associated weights and background predictions and associated weights with \code{NA}s.}

\item{...}{Other arguments (unused).}
}
\value{
Named numeric vector.
Fielding, A.H. and J.F. Bell. 1997. A review of methods for the assessment of prediction errors in conservation presence/absence models. \emph{Environmental Conservation} 24:38-49. \doi{10.1017/S0376892997000088}
}
\description{
This function is similar to the \code{\link[predicts]{threshold}} function in the \pkg{predicts} package, which calculates thresholds to create binary predictions from continuous values. However, unlike that function, it allows the user to specify weights for presences and absence/background predictions. The output will thus be the threshold that best matches the specified criterion taking into account the relative weights of the input values.
}
\examples{
set.seed(123)

# set of bad and good predictions at presences
bad <- runif(100)^2
good <- runif(100)^0.1
hist(good, breaks=seq(0, 1, by=0.1), border='green', main='Presences')
hist(bad, breaks=seq(0, 1, by=0.1), border='red', add=TRUE)
pres <- c(bad, good)
contrast <- runif(1000)
evalThreshold(pres, contrast)

# upweight bad predictions
presWeight <- c(rep(1, 100), rep(0.1, 100))
evalThreshold(pres, contrast, presWeight=presWeight)

# upweight good predictions
presWeight <- c(rep(0.1, 100), rep(1, 100))
evalThreshold(pres, contrast, presWeight=presWeight)
}
\seealso{
\code{\link[predicts]{threshold}}, \code{\link[predicts]{pa_evaluate}}, \code{\link{evalAUC}}, \code{\link{evalMultiAUC}}, \code{\link{evalContBoyce}}, \code{\link{evalThresholdStats}}, \code{\link{evalTjursR2}}, \code{\link{evalTSS}}
}
